;-*- Mode: Common-lisp; Package: nisp; Readtable: ytools; -*-
(in-package :nisp)
;;; $Id: litlisp.lisp,v 1.44 2006/05/18 18:51:55 dvm Exp $

(depends-on %module/ nisp)

(depends-on :at-run-time %wtools/ levcount %ytools/ debug)

;;;;(depends-on %ytools/ multvalhacks)

(depends-on (:at :slurp-time) %ytools/ bq)

(eval-when (:compile-toplevel :load-toplevel :execute :slurp-toplevel)
   (export '(litlisp tex-litlisp file-seg code-seg ignore-seg
	     insert-seg skip-seg tangling code-file
	     source-file use-frag omit-frag define-frag
	     referring-paper declare-this-paper)))

;;; Builds on txtlisp to support literate programming, by allowing
;;; extra "code" files to be declared and, when requested, output.
;;; Documentation is now located in 
;;; ~/UNIfied/doc/litlisp/litlisp-doc.txt

;;; The difference between a "fragment" and a "segment": The former are
;;; extracted from preexisting source files.  The latter are used to 
;;; assemble new source files.

(defnisptype Prog-segment 
   (Structure 
      id -  Symbol defn - (~ String) used defined - Boolean
    ;; --'defined' is true only if it's defined in the input file,
    ;;   even if it was retrieved from the .seg file .
         (:handler
	      (print-object - Void (seg - Prog-segment srm - Stream)
		 (let ((str (and (!_defn seg)
				 (string-subseq
				    (!_defn seg)
				    0
				    (min (string-length (!_defn seg))
					 10)))))
		    (out (:to srm)
		       "#<Prog-segment " (!_id seg)
			     (:q ((not (!_defined seg)) "?"))
			     (:q ((!_used seg) " used")
				 (t " unused"))
			     (:q ((and str
				      (not (string= str "")))
				  str))
			     ">"))))))

(defnisptype Prog-fragment
       (Structure name - Symbol
		  ;; ID of file this came from --
		  source-id - (~ Symbol)
		  defined - Boolean
		  ;; -- false until it's encountered in the file.
		  ;; Retrieval from .lux file isn't enough, because
		  ;; the definition might have been deleted.
		  recalled-state state
		     - (Const :created :used :omitted :unused)
		  line-num - Integer
		  subfrags - (Lst (Lrcd Integer Prog-fragment))
		  ;; -- Pairs of (position fragment) pairs.
		  contents - (~ String)
	  (:handler
	      (print-object - Void (frag - Prog-fragment srm - Stream)
		 (let ((str (and (!_contents frag)
				 (string-subseq
			            (!_contents frag)
				    0
				    (min (string-length (!_contents frag))
					 10))))
		       (numsubs (length (!_subfrags frag))))
		    (let-fun ((:def state-string (state)
				 (case state
				    (:used "used")
				    (:unused "unused")
				    (:omitted "omitted")
				    (t "unmentioned"))))
		       (out (:to srm)
			  "#<Prog-fragment " (!_name frag)
			        (:q ((not (!_defined frag)) "?"))
				" File: " (!_source-id frag)
				" [" (:a (state-string (!_recalled-state frag)))
				     "><"
				     (:a (state-string (!_state frag)))
				"] "
				(:q ((> numsubs 0) "[" numsubs " subfrags] "))
				(:q ((or (not str)
					 (not (string= str "")))
				     str))
				">"))))))
  (conser (name - Symbol source-id - (~ Symbol) defined - Boolean
	   recalled-state state
	     - (Const :created :used :omitted :unused)
	   line-num - Integer
	   contents - (~ String))
     (make ^^ name source-id defined recalled-state state
	      line-num !() contents)))

;;; A Prog-fragment under construction --
(defnisptype Constructing-frag
	     (Structure name - Symbol
			source-id - Symbol
			;; -- :contrived if created by 'define-frag',
			;; and hence associated with the .txl file,
			;; not a separate source file.
			at-start - Boolean
			;; -- At start of whole fragment, not a line
			stream - (~ Stream)
			;; -- It's false for the top level of a file
			;; all of whose characters are to be discarded,
			;; except those that occur in marked fragments.
			line-num - Integer
			subfrags - (Lst (Lrcd Integer Prog-fragment))))

(declare-txtlisp-vars
   this-paper-name - (~ String)
   seg-table - (Lst (Lrcd Symbol Prog-segment))
   seg-stream -  Stream ; to file containing permanent seg table.
   changed-segs -  (Lst Symbol)  ; ids of segs that changed
   source-files - (Lst (Lstructure ()
			   id - Symbol
			   info - (Lstructure ()
				      filename - String
   other-info - Sexp)))
   ;; -- The Sexp gives the location of the file.
   fragment-table - (Lst (Lrcd Symbol Prog-fragment))
   changed-frags undefined-frags -  (Lst Symbol)  ; ids of frags that changed
   tangling -  Boolean   ; true if generating code files
     ;; --"Tangling" is term from Knuth's original WEB hack.
   litlisp-out-files - (Lst (Lrcd Symbol (~ Pathname) Stream))
     ;; -- For each code (and other) file declared, its pathname and stream
     ;; If pathname is false, then stream was opened by someone else,
     ;; and should not be closed here.
   code-stream -  Stream ; the stream to the code file currently being
                         ;  generated
   ;; Some callers will have already opened a stream for one of the
   ;; code files. --
   anticipated-code-file - (~ (Lrcd Pathname Stream))
   ;; If so, use this flag to test if the anticipated stream was used --
   used-anticipated-code-file - Boolean
   central-out-stream - Stream
   ;; If true, "Insert-seg" must include a page reference to the segment.--
   pagerefs-to-code-segs - Boolean
   ;; 'true' if we've already output the definition of the \segadjust variable.
   seg-vspace-adjust-defined - Boolean
 )

(specdecl seg-pathname* - Pathname)

(!= seg-pathname* (make-pathname :type "seg"))

(specdecl post-file-transduce-hooks* - (Lst (Fun Void ())))

(specdecl (seg-delimiter* ",|_")
                  ;;;;   "~@~" <- no good because it starts with a "guarded" char.
;;;;	  (cxt-starter* ":.")
;;;;	  (cxt-ender* ".:")
;;;;	  ;; This version causes cxt marks to be suppressed
;;;;	  ;; in text output; must be same length as cxt-ender* --
;;;;	  (cxt-ender-hide* "./")
	  - String
	  (litlisp-repeat-til-stable* true)
	  - Boolean
	  (litlisp-num-runs* 5)
	  num-changed-segs* num-changed-frags* - Integer)

(defproc litlisp - (~ Pathname)
                   (in-file out-file - Obj
		    mode-desig - (Alt Symbol (Fun Txtlisp-mode (Obj)))
		    vars - (Lst (Lrcd Symbol Obj)))
   (bind ((num-changed-segs* 0)
	  (num-changed-frags* 0)
	  (txtlisp-control (decipher-mode-from-desig mode-desig !())))
      (repeat :for ((i = 1 :by 1) (num-attempts litlisp-num-runs*))
       :until (and (> i num-attempts)
		   (repeat :for ((num-more nil) - (~ Integer))
		     (out (:to *query-io*)
			"Cycled " (- i 1) " times without segments stabilizing;"
			" how many more? ")
		     (!= num-more (in (:from *query-io*) :object))
		   :until (and (is Integer num-more)
			       (>= num-more 0))
		   :result (cond ((> num-more 0)
				  (!= num-attempts (+ *-* num-more))
				  false)
				 (t true))))
	 (txtlisp in-file out-file
		  txtlisp-control
		  (cons (tuple 'tangling true)
			vars))
;;;;         (breakpoint "num-changed-segs* = " num-changed-segs*)
       :until (or (not litlisp-repeat-til-stable*)
		  (= num-changed-segs* 0))
         (out (:to *error-output*)
	      :% "*** File will be rerun until segments stop changing ***"
	      :%))))

(defproc declare-this-paper - Void (id - String)
   (cond ((and (tv-is-bound this-paper-name)
	       (tv-val this-paper-name)
	       (not (string= (tv-val this-paper-name)
			     id)))
	  (err-out "Warning: changing the declared name of this paper"
		   :% " from " (tv-val this-paper-name) " to " id)))
   (!= (tv-val this-paper-name) id))

(defproc code-file - Void (id - Symbol filespec - Obj)
   (cond ((ensure-litlisp-env)
	  (cond ((and (tv-is-bound anticipated-code-file)
		      (eq id (tv-val anticipated-code-file)))
		 (!= (tv-val litlisp-out-files)
		     (cons (tuple id
				  (be (Const nil) false)
				  (tv-val central-out-stream))
			   (tv-val litlisp-out-files)))
		 (cond ((tv-is-bound used-anticipated-code-file)
			(!= (tv-val used-anticipated-code-file) true))))
		(t
		 (note-litlisp-out-file id filespec))))
	 (t
	  (note-litlisp-out-file id filespec))))

(specdecl (source-file* false) - (~ Pathname))

(defproc source-file - Void (id - Sexp filespec - Obj)
   (cond ((or (is String id)
	      (and (not (is Symbol id))
		   (progn
		      (signal-problem source-file
			"Source file id is not a symbol or string: " id
			:% "  [denotes " filespec "]"
			(:proceed "I will force it to be a symbol using"
				  " 'intern'"))
		      (!= id (out (:to :string) (:a *-*)))
		      true)))
	  (!= id (intern *-*))))
   (cond ((not id)
	  (signal-problem source-file
	     "Illegal to use 'nil' as the id for a source file "
	     "[denoting " filespec "]")))
   (decl (id - Symbol)
      (ensure-litlisp-env)
      (ensure-fragment-table)
      ;; Also bind tv-val this-paper-name if not bound already --
      (cond ((not (tv-is-bound this-paper-name))
	     (!= (tv-val this-paper-name)
		 false)))
      (bind ((source-file* (->pathname filespec)))
	 (with-open-file (srm source-file* :direction ':input)
	    (source-stream-extract-fragments
		srm
		(list (make Constructing-frag
			    id id false false 0 !()))
		"source file " id)))))

(defnisptype Referring-paper
      (Structure label - String loc - Sexp default - Boolean))

;;; Papers that refer to fragments in this source file. 
;;; Second element is true iff paper is a "default," meaning
;;; that it does not have to be mentioned explicitly in fragments.
(specdecl (referring-papers* !())
	  - (Lst (Lrcd String Referring-paper)))

(defproc source-stream-extract-fragments - Void
	                (srm - Stream
			 frag-stack  - (Lst Constructing-frag)
			 dbg-string - String id - Symbol)
   (let ((stack-depth (len frag-stack)))
      (bind ((referring-papers* !())
	     - (Lst (Lrcd String Referring-paper))
	     (current-source-stream* srm))
	 (out-indent *error-output* 3
	    (dbg-out txtlisp-dbg*
		"Gathering code fragments from " (:a dbg-string)
		id (:q (source-file* " = " source-file*)))
	    (cond ((not (txtlisp-var-is-bound 'this-paper-name))
		   (err-out "Warning -- txtlisp var 'this-paper-name'"
			    " is not bound."
			    :% " Insert ~~ (declare-this-paper <namestring>)"
			    " <string>) to set it.")))
	    (unwind-protect
	       (stream-lines-extract-fragments srm id frag-stack)
	      ;; Nonlocal exit can happen only if error occurred,
	      ;; so don't try to make the Prog-fragments look good,
	      ;; just close their streams --
	      (repeat :for ((frag :in frag-stack)
			    (n = (len frag-stack)
			       :by -1 :to (+ stack-depth 1)))
	       ;; Leave streams passed by caller open.
		 (cond ((!_stream frag)
			(close (!_stream frag))))))))))

(defproc stream-lines-extract-fragments - Void
	                (srm - Stream id - Symbol
			 frag-stack - (Lst Constructing-frag))
   (repeat :for (line - String
		 (line-num = 1 :by 1) - Integer
		 last-line - Boolean
		 (control-lines !()) - (Lst String)
		 (ignored-frag-stack !()) - (Lst Symbol))
       (!= < line last-line > (read-line srm false eof*))
    :until (eq line eof*)
       (multi-let (((frag-control rest-of-line)
		    (line-parse-for-frag-control line)))
	  (cond (frag-control
		 (cond ((memq frag-control
			      '(:begin :end))
			(handle-frag-boundary
			   rest-of-line
			   (eq frag-control ':begin)))
		       ((eq frag-control ':read-for-eval)
			(cond ((and (not (null frag-stack))
				    (not (!_stream
					     (head frag-stack))))
			       (on-list rest-of-line
					control-lines))
			      (t
			       (signal-problem
				  source-stream-extract-fragments
				  "Control line inside fragment"
				  " -- fragment"))))
		       (t
			(signal-problem
			    source-stream-extract-fragments
			    "Fumbled " frag-control))))
		(t
		 (cond ((not (null control-lines))
			(control-lines-eval)))
		 (let* ((cfrag (head frag-stack))
			(cstream (!_stream cfrag)))
		    (cond (cstream
			   (new-line-unless-at-start cfrag)
			   (cond ((eq (!_name cfrag)
				      ':eval)
				  (out (:to cstream)
				       (:a line)))
				 (t
				  (textify line cstream)))))))))
    :until last-line
    :result nil
;;;;                    (cond ((not (null frag-stack))
;;;;			       (signal-problem source-file
;;;;				  "Source file ends with fragments still"
;;;;				  " being constructed: "
;;;;				  (<# !_name frag-stack))))
    :where

 (:def handle-frag-boundary - Void
		       (rest-of-line - String
			begin - Boolean)
	  (multi-let (((for-this-paper fragname)
		       (extract-frag-name rest-of-line begin)))
	     (cond (for-this-paper
		    (cond (begin
			   (frag-stack-push fragname))
			  (t
			   (frag-stack-pop fragname))))
		   (t
		    ;; must be a 'begin'
		    (push fragname ignored-frag-stack)))))

 (:def frag-stack-push - Void (name - Symbol)
    (let ((cfrag (head frag-stack)))
       (cond ((!_stream cfrag)
	      (new-line-unless-at-start cfrag)))
       (stream-indent *error-output* 3)
       (dbg-out txtlisp-dbg*
	    "Gathering code fragment " name)
       (push (make Constructing-frag
		     name id true
		     (make-string-output-stream)
		     line-num !())
	     frag-stack)))

 (:def frag-stack-pop - Void (name - Symbol)
	 (let ((ignoring (memq name ignored-frag-stack)))
	    (repeat :for ((frag :in frag-stack))
	     :result (cond (ignoring
			    (ignore-pop name))
			   (t
			    (signal-problem source-file
			       "End marker for fragment "
			       name " -- with no beginning"
			       (:proceed !"I'll ignore the ~
					 end marker"))))
	     :until (string= (!_name frag)
			     name)
	     :result
		(progn
		   (cond (ignoring
			  (signal-problem source-file
			     "Ignored fragment and kept fragment"
			     " have the same name: " name)))
		   (!= frag-stack
		       (pop-to-fragment frag-stack frag))))))

 (:def ignore-pop - Void (name - Symbol)
	 (cond ((and (not (null ignored-frag-stack))
		     (eq (head ignored-frag-stack)
			 name))
		(pop ignored-frag-stack))
	       (t
		;; Don't worry too much about stack
		;; discipline
		(err-out "Warning: end of fragment "
			 name " out of place")
		(!= ignored-frag-stack
		    (delete name *-*)))))

 (:def control-lines-eval - Void ()
	  (let ((control-input
		   (out (:to :string)
		      (:e (repeat :for ((line :in control-lines))
			     (:o (:a line) 1))))))
	     (with-input-from-string
			(ssrm control-input)
		(repeat :for ((r = (read ssrm false eof*)
				 :then :again))
		 :until (eq r eof*)
		    (eval r)))))

 (:def new-line-unless-at-start - Void (cfrag - Constructing-frag) 
   (cond ((!_at-start cfrag)
	  (!= (!_at-start cfrag) false))
	 (t
	  (out (:to (!_stream cfrag))
	       (:q ((eq (!_name cfrag) ':eval)
		    :%)
		   (t
		    (:e (let ((char-guard (char-guarder txtlisp-control*))
			      - (Fun (~ String) (Char)))
			   (let ((s (funcall char-guard #\Newline)))
			      (:o (:q (s (:a s))
				      (t :%))))))))))))

;;;; ;; Return true iff we're actually extracting a fragment --
;;;; (:def maybe-new-line - Boolean ()
;;;;	  (cond ((null frag-stack)
;;;;		 (signal-problem source-stream-extract-fragments
;;;;		    "Fragment stack blown"))
;;;;		(t
;;;;		 (let ((cfrag (head frag-stack)))
;;;;		    (cond ((!_stream cfrag)
;;;;			   (new-line-unless-at-start
;;;;			       (!_at-start cfrag)
;;;;			       (!_stream cfrag))
;;;;			   (!= (!_at-start cfrag) false)
;;;;			   true)
;;;;			  (t false))))))
 ))

(defproc ensure-fragment-table - Void ()
   (cond ((not (tv-is-bound fragment-table))
	  (err-out "fragment table doesn't exist; creating")
	  (create-fragment-table))))

(defproc create-fragment-table - Void ()
	  (!= (tv-val fragment-table) !())
	  (!= (tv-val changed-frags) !())
	  (!= (tv-val undefined-frags) !())
	  ;;;(note-txtlisp-aux-form '(!= (tv-val fragment-table) !()))
	  (after-file-transduction
	     (let ((chfragl (tv-val changed-frags))
		   - (Lst Symbol))
		(!= num-changed-frags* (len chfragl))
		(cond ((> num-changed-frags* 0)
		       (out (:to *error-output*)
			    "Changed fragments: " (reverse chfragl) :%))))
	     (let ((unfragl (tv-val undefined-frags)))
	        (cond ((> (len unfragl) 0)
		       (out (:to *error-output*)
			    "Fragments never defined: " unfragl))))
	     (fragment-table-to-lux-file)
	     (let ((frag-tab (tv-val fragment-table))
		   - (Lst (Lrcd Symbol Prog-fragment)))
	        (let ((frag-trans-forms
			 (repeat :for ((e :in frag-tab))
			   :within
			     (let ((e-subs (!_subfrags (second e))))
				(:continue
				 :when (not (null e-subs))
				 :collect
				   !`1(!= (!_subfrags
					    (find-frag ',1(!_name (second e))))
					  (<# pos-name-subfrag-entry
					      ',1(<# first e-subs)
					      ',1(<# (\\ (e)
							(!_name (second e)))
						     e-subs))))))))
		   (cond ((not (null frag-trans-forms))
			  (note-txtlisp-aux-form
			     `(decl ()
				  ,@frag-trans-forms))))))))

;;;;	        (repeat :for ((e :in frag-tab))
;;;;		   (let ((e-subs (!_subfrags (second e))))
;;;;		      (note-txtlisp-aux-form
;;;;			 !`1(!= (!_subfrags
;;;;				   (find-frag ',1(!_name (second e))))
;;;;				(<# pos-name-subfrag-entry
;;;;				    ',1(<# first e-subs)
;;;;				    ',1(<# (\\ (e)
;;;;					      (!_name (second e)))
;;;;					   e-subs)))))))))))

(deffunc pos-name-subfrag-entry - (Lrcd Integer Prog-fragment)
	           (position - Integer frag-name - Symbol)
   (tuple position (find-frag frag-name)))

(specdecl (frag-begin* "<<<<")
	  (frag-end* ">>>>")
	  (frag-control* "^^^^")
	  - String)

;;; Returns < b, sort, rest-of-line >
;;; where b is true if this contains a frag-control mark;
;;; sort is :begin, :end, or :read-for-eval
(deffunc line-parse-for-frag-control
                   - (Mlv (Const :begin :end :read-for-eval nil)
			  (~ String))
	     (line - String)
   (let-fun ()
      (control-nest
         (multi-let (((found after)
		      (string-after-if-occurs frag-begin* line)))
	    (cond (found
		   (values ':begin after))
		  (t :try-next)))
       :try-next
         (multi-let (((found after)
		      (string-after-if-occurs frag-end* line)))
	    (cond (found
		   (values ':end after))
		  (t :try-next)))
       :try-next
         (multi-let (((found after)
		      (string-after-if-occurs frag-control* line)))
	    (cond (found
		   (values ':read-for-eval after))
		  (t
		   (values false false)))))

    :where
			   
      ;; Return < true, sub > if string 'looking-for' occurs in string
      ;; 'looking-in' and 'sub' is the rest of 'looking-for' after its first
      ;; occurrence.  Otherwise, return < false, don't-care >.
      (:def string-after-if-occurs - (Mlv Boolean (~ String))
	                       (looking-for looking-in - String)
	   (let ((pos (search looking-for looking-in))
		 - (~ Integer))
	      (cond (pos
		     (values true
			     (string-subseq
			        looking-in
				(+ pos (string-length looking-for))
				(string-length looking-in))))
		    (t
		     (values false false)))))))

;;; First value returned is true if frag-name is to be heeded.
;;; For endings, it's always true, because whether it's heeded or
;;; or not depends on what frag beginnings are on which stack. --
(deffunc extract-frag-name - (Mlv Boolean Symbol)
	            (line - String begin - Boolean)
   (let-fun ()
      (multi-let (((fragname after-pos)
		   (read-from-string
			   line false eof*)))
	 (cond ((eq fragname eof*)
		(signal-problem extract-frag-name
			  "Can't find name of fragment in line"
			  :% line :%))
	       ((is Symbol fragname)
		(cond (begin
		       (let ((papername (read-from-string
					   line false eof*
					   :start after-pos)))
		       ;;;;		   (dbg-save fragname papername)
			  (values (for-this-paper papername fragname)
				  fragname)))
		      (t
		       (values true fragname))))
	       (t
		(signal-problem extract-frag-name
		   "Fragment has illegal name " fragname
		   " -- should be a symbol"))))

    :where

       (:def for-this-paper - Boolean (x - Obj fragname - Symbol)
	  (cond ((and (tv-is-bound this-paper-name)
		      (tv-val this-paper-name))
		 (cond ((eq x eof*)
			;; No declaration.  Check if it's implicit
			(cond ((eq fragname ':eval)
			       ;; Unconstrained :eval's are always
			       ;; done
			       true)
			      (t
			       (let ((rp (alref referring-papers*
						(tv-val this-paper-name)
						false
						:test #'string=)))
				  (cond (rp
					 (!_default rp))
					(t false))))))
		       ((is String x)
			(string= x (tv-val this-paper-name)))
		       ((is (Lst String) x)
			(member= #'string=
				 (tv-val this-paper-name)
				 x))
		       (t
			(dbg-save :run-loud x)
			(signal-problem extract-frag-name
			   "Can't find name of referring paper in"
			   :% line :%))))
		(t true)))))

(defproc pop-to-fragment - (Lst Constructing-frag)
	          (frag-stack - (Lst Constructing-frag)
		   frag - Constructing-frag)
   (repeat :for ((s = (pop frag-stack) :then :again)
		 - Constructing-frag)
      (cond ((eq (!_name s) ':eval)
	     (eval (read-from-string
		      (get-output-stream-string (!_stream s)))))
	    (t
	     (let ((prog-frag (note-frag s)))
		(close (!_stream s))
		(dbg-out txtlisp-dbg*
		   "Finished fragment " (!_name s))
		(stream-indent *error-output* -3)
		(cond ((null frag-stack)
		       (signal-problem pop-to-fragment
			  "Frag-stack blown"))
		      (t
		       (let* ((next-frag (head frag-stack))
			      (frag-stream (!_stream next-frag)))
			  ;; Inner fragment
			  (cond ((and prog-frag frag-stream)
				 (!= (!_subfrags next-frag)
				     (cons (tuple (file-position frag-stream)
						  prog-frag)
					   *-*))))))))))
    :until (eq s frag)
    :result frag-stack
      (err-out "fragment " (!_name s)
	       " ends prematurely [" (!_name frag) "]")
    :until (null frag-stack)
    :result (signal-problem pop-to-fragment
	       "Fumbled fragment " frag)))

;;; Returns the fragment corresponding to 's' (with the same name).
;;; If 's' has name '_', a new Prog-fragment is always created --- and
;;; purely for debugging purposes, because it will always be omitted.
;;; Cancel that.  To make it work right, we would have to record these
;;; fragments in the .lux file, which is absurd.
(defproc note-frag - (~ Prog-fragment) (s - Constructing-frag)
   (let* ((contents 
	     (get-output-stream-string
		(!_stream s)))
	   - String
	  (frag (cond ((eq (!_name s) '_)
		      false)
		     (t
		      (place-frag (!_name s) (!_source-id s)
				  contents))))
	    - (~ Prog-fragment))
      (cond (frag
	     (let-fun ()
		(cond ((eq (!_state frag)
			   ':created)
		       (!= (!_defined frag) true)
		       (!= (!_line-num frag) (!_line-num s))
		       (!= (!_state frag) ':unused)
		       (xfer-subfrags))
		      ((!_defined frag)
		       (dbg-save :run-loud frag s)
		       (signal-problem note-frag
			      "fragment " (!_name s) " defined twice: "
			      :% "<1>  [from " (!_source-id frag) "]"
				 (!_contents frag)
			      :% "<2> [from " (!_source-id s) "]"
				 contents
			  (:proceed !"I'll use only the first, ~
				      which may therefore appear ~
				      in more than one place")))
		      (t
		       (!= (!_defined frag) true)
		       (cond ((and (!_contents frag)
				   (not (string= (!_contents frag)
						 contents)))
			      (on-list (!_name frag)
				       (tv-val changed-frags))))
		       (!= (!_line-num frag) (!_line-num s))
		       (!= (!_contents frag) contents)
		       (xfer-subfrags)))
	      :where
	         (:def xfer-subfrags ()
		    (!= (!_subfrags frag)
			(!_subfrags s))))))
      frag))

;;;;(defproc maybe-fresh-line - Void
;;;;	             (next-frag - Constructing-frag)
;;;;   (new-line-unless-at-start (!_at-start next-frag)
;;;;			     (!_stream next-frag)))

;;; True if in the middle of 'code-seg' or 'file-seg'.
;;; In which case, *standard-output* is going to a temporary string
;;; that will be transmogrified into code and text versions.
(specdecl (building-segment* false) - (~ (Lrcd String Symbol)))

(defproc insert-seg - Void (name - Symbol)
      (ensure-litlisp-env)
      (cond (building-segment*
	     (out (:a seg-delimiter*) name (:a seg-delimiter*))
	     (let ((e (val-or-initialize
		         (alref (tv-val seg-table)
				name)
			 :init (progn
				  (out (:to *error-output*)
				       "Undefined segment: " name :%)
				  (make Prog-segment
				     name (be (Const nil) false)
				     false false))))
                   - Prog-segment)
		(!= (!_used e) true)))
	    (t
	     (out (:to *error-output*)
		"insert-seg at top level (segment name " name ")" :%)
	     (out "<<Insert: " name ">>"))))

(defproc skip-seg - Void (&rest _)
   nil)

;;; Set to true to make <<Define <id> ... >> appear around
;;; fragments.
(specdecl (frag-show-define* true) - Boolean)
;;; -- Why would this ever be set to false??

;;; 'source' is the id of a source file, or false if you think
;;; it will always be clear which file this fragment comes from,
;;; even if you come across this .txl file five years from now.
;;; 'omit' is a list of names of subfragments to omit.
;;; 'replace' is a list of pairs (current repl) of fragment names. 
;;; The occurrence of subfragment 'current' is replaced by fragment 'repl'.
;;; If 'repl' is ':omit', that's the same as putting it in the 'omit' list.
;;; But 'repl' can also be a nonatomic S-expression, in which case
;;; it is interpreted as a recursive call to 'use-frag' (so you can
;;; have replacements within replacements).
;;; 'recursive' is not for users; it keeps track of whether we're at
;;; the top level of a nest of fragments.  If so, we need to wrap some
;;; boilerplate around all the output.
(specdecl use-frag (Fun Void (Symbol . Sexp)))
(defun use-frag (name &key (show-define frag-show-define*)
			   source
			   multi-okay omit replace
			   recursive)
   (decl (name - Symbol multi-okay - Boolean
	  omit - (Lst Symbol) replace - (Lst (Lrcd Symbol Sexp))
	  recursive - Boolean)
      (let ((frag (place-frag name source false)))
	 (cond ((eq (!_state frag) ':created)
		(signal-problem use-frag
		   "Fragment " name " not defined by any source file"
		   (:proceed "I will define it as an empty fragment"))
		(!= (!_state frag) ':used))
	       ((eq (!_state frag) ':used)
		;; This is not always an error; you can suppress the warning
		;; with ':multi-okay true'
		(cond ((not multi-okay)
		       (err-out "Warning -- fragment " name
				" used more than once"))))
	       (t
		(!= (!_state frag) ':used)))
	 (multi-let (((preamble postamble - String)
		      (cond ((or recursive
				 (not show-define))
			     (values "" ""))
			    (t
			     (embedded-code-pre-post-strings
			        (!_name frag)
			        (cond ((!_source-id frag)
				       (out (:to :string)
					  (!_name frag)
					  (:q ((not (eq (!_source-id frag)
							':contrived))
					       " [" (:a (!_source-id frag))
						    ", line " (!_line-num frag)
					       "]"))))
				      (t (!_name frag)))
				false)))))
	    (out (:a preamble))
	    (cond ((!_contents frag)
		   (frag-out ;;;;(!_name frag)
			     (!_contents frag)
			     (!_subfrags frag)
			     (append replace
				     (<# (\\ (s) (tuple s ':omit))
					 omit))))
		  (t
;;;;		   (new-line-unless-at-start at-start *standard-output*)
		   (out "<< Undefined fragment " name " >>")
		   false))
	    (out (:a postamble))))))

;;; Returns true iff at the beginning of a line when finished
(defproc frag-out - Void (    ;;;; name - Symbol
			  contents - String
			  subfrags - (Lst (Lrcd Integer Prog-fragment))
			  replace - (Lst (Lrcd Symbol Sexp)))
   (!= subfrags (sort (list-copy *-*)
		     (\\ (s1 s2 - (Lrcd Integer Prog-fragment))
			 (< (first s1) (first s2)))))
   (multi-let ((frag-ref-produce (frag-ref-producer txtlisp-control*))
		    - (Fun String (Symbol Boolean)))
      (repeat :for ((sub :in subfrags) - (Lrcd Integer Prog-fragment)
		    ;; Beginning of current snippet --
		    (snip-beg 0) - Integer)
       :result (repeat :for ((i = snip-beg
				:to (- (string-length contents) 1)))
		  (out (:a (elt contents i))))
	 (repeat :for ((i = snip-beg :to (- (first sub) 1)))
	    (out (:a (elt contents i))))
	 ;; Now output subfrag or its replacement --
	 (let* ((sub-frag (second sub))
		(frag-name (!_name sub-frag))
		(frag-contents (!_contents sub-frag))
		(actual-sub-name (alref replace frag-name)))
	    (cond ((eq actual-sub-name ':omit)
		   (out (:a (code-seg-ellipsis txtlisp-control*))))
		  (actual-sub-name
		   (apply #'use-frag
			  `(,@(cond ((atom actual-sub-name)
				     (list actual-sub-name))
				    (t actual-sub-name))
			    :recursive true)))
		  (t
		   ;; No replacement, 
		   (let ((state (!_recalled-state sub-frag)))
		      (ecase state
			 (:unused
;;;;			  (setq unused-sub-frag* sub-frag)
;;;;			  (maybe-new-line)
;;;;			  (out (:a (funcall preamble frag-name))
;;;;			       :%)
			  (cond ((string= frag-contents "")
				 (err-out "Empty fragment: "
					  frag-name))
				(t
				 (frag-out   ;;;; frag-name
					   frag-contents
					   (!_subfrags sub-frag)
					   !())))
;;;;			  (out (:a (funcall postamble frag-name))
;;;;			       :%)
			  )
			 (:used
;;;;			  (maybe-new-line)
			  (out (:a (funcall frag-ref-produce
					    frag-name (!_defined sub-frag)))))
			 (:omitted))))))
	 (!= snip-beg (first sub))
;;;;       :where
;;;;	  (:def maybe-new-line - Void ()
;;;;	      (new-line-unless-at-start at-start *standard-output*)
;;;;	      (!= at-start false))
	  )
;;;;      (out (:a (funcall postamble name)))
      ))

#|
(defproc new-line-unless-at-start - Void (at-start - Boolean stream - Stream)
   (cond ((not at-start)
	  (let ((char-guard (char-guarder txtlisp-control*))
		- (Fun (~ String) (Char)))
	     (let ((s (funcall char-guard #\Newline)))
		(out (:to stream)
		     (:q (s (:a s))
			 (t :%))))))))
|#

(defproc omit-frag - Void (name - Symbol)
   (let ((frag (place-frag name false false)))
      (!= (!_state frag) ':omitted)))

(defproc place-frag - Prog-fragment (name - Symbol source-id - (~ Symbol)
				     contents - (~ String))
   (let ((frag (val-or-initialize
		  (alref (tv-val fragment-table)
			 name)
		  :init (progn
			   (err-out "Undefined fragment " name
				    " -- creating")
			   (make Prog-fragment name source-id false
					      ':unused ':created
					      0 contents))))
         - Prog-fragment)
      (cond ((and source-id (not (eq (!_source-id frag) source-id)))
	     (signal-problem place-frag
		"Fragment " name " has been associated with two different"
		:% " source files: " (!_source-id frag) " and " source-id
		(:proceed "I will use the id " source-id))
	     (!= (!_source-id frag) source-id)))
      frag))

(deffunc find-frag - Prog-fragment (name - Symbol)
   (or (alref (tv-val fragment-table) name)
       (signal-problem find-frag
	  "Undefined fragment " name)))

;;; By analogy with #= and ##.
;;; !#= f    =>  (code-seg 'f)
;;; !#== f   =>  (file-seg 'f)
;;; !## f    =>  (insert-seg 'f)
;;; !#> s    =>  (define-frag 'f)
;;; !#< s    =>  (use-frag 'f)
;;; !#(= is a synonym for !#= and !#(> for !#>.
;;; (Actually, you can put as many open parens as you want after the first #
;;; for any of the constructs, but it doesn't make much sense for any but !#=
;;; and !#>.)
;;; The paren versions can be terminated by ~~).
(def-excl-dispatch #\# (srm _)
   :readtable txl-readtable*
   (repeat :while (char= (in (:from srm) :peek) #\()
      (in (:from srm) :char)
;;;;      (out (:to *query-io*) "." (:a ))
      )
   (let ((ch (in (:from srm) :char)))
      (expect-sym-as-arg
	  srm ch
	  (case ch
	     (#\=
	      (cond ((char= (in (:from srm) :peek)
			    #\=)
		     (in (:from srm) :char)
		     'file-seg)
		    (t
		     'code-seg)))
	     (#\# 'insert-seg)
	     (#\> 'define-frag)
	     (#\< 'use-frag)
	     (t (signal-problem :place "!#..."
		    "Illegal character " ch " after !#"
		    (:prompt-for "intended character " #\#)))))))

(defproc expect-sym-as-arg - Sexp (srm - Stream char - Char
				   fcn-name - (Const file-seg code-seg insert-seg
						  define-frag use-frag))
      (let ((name (read-preserving-whitespace srm)))
	 (cond ((is-Symbol name)
		`(,fcn-name ',name))
	       (t
		(signal-problem expect-sym-as-arg
		   "!#" (:a char)
;;;;		   (:q ((eq fcn-name 'code-seg)
;;;;			"!#=")
;;;;		       (t
;;;;			"!##"))
		   " followed by a nonsymbol: " name)))))

;;; The trick with 'code-seg' and 'file-seg' is that they must generate one
;;; version in the code file and another in the text output file.
;;; So we generate an intermediate version in a string and then spew it in
;;; two different directions.

(specdecl (break-on-code-seg-change* false) - Boolean)

(defproc code-seg - Void (name - Symbol)
   (ensure-litlisp-env)
   (let ((code-stream (file-name-stream name)))
      (cond (code-stream
	     ;; Tricky way to get to file-seg --
	     (code-stream-gets-seg code-stream name))
	    (t
	     (create-code-seg name)))))

(defproc create-code-seg - Void (name - Symbol)
   (multi-let (((code-file-v text-file-v - String)
		(transduce-to-code-and-text `("code" ,name) false)))
      (let ((e (val-or-initialize
		  (alref (tv-val seg-table) name)
		  :init (make Prog-segment name false false false)))
	    - Prog-segment)
	 (cond ((!_defined e)
;;;;		(dbg-save e code-file-v text-file-v)
		(signal-problem code-seg
		   "Code segment " name " defined more than once"
		   (:continue "I will ignore all but the first definition")))
	       (t
		(!= (!_defined e) true)
		(out (:to (tv-val seg-stream))
		   "~~" name :%
		   (:a code-file-v)
		   :%       ;;;; <-  Should be ignored when re-read
		   "~~" name :%)
		(cond ((or (not (!_defn e))
			   (not (string= (!_defn e) code-file-v)))
		       (cond (break-on-code-seg-change*
			      (dbg-save :run-loud e code-file-v)
			      (breakpoint code-seg
				 "Changing: " (!_id e))))
                       (on-list name (tv-val changed-segs))
		       (!= (!_defn e) code-file-v))))))
	 (multi-let (((preamble postamble)
		      (embedded-code-pre-post-strings name name false)))
	    (out (:a preamble)
		 :%
		 (:a text-file-v)
		 (:a postamble)
		 :%))))

(defproc file-seg - Void (filespec - Obj)
   (let ((code-stream (and (ensure-litlisp-env)
			   (or (file-name-stream filespec)
			       (signal-problem file-seg
				  "Undeclared or absurd output file " filespec)))))
      (code-stream-gets-seg code-stream filespec)))

(defproc code-stream-gets-seg - Void (code-stream - (~ Stream)
				      seg-spec - Symbol)
      (!= (tv-val code-stream)
	  code-stream)
      (multi-let (((code-file-v text-file-v - String)
		   (transduce-to-code-and-text `("file" ,seg-spec) false)))
 	 (cond (code-stream
		(out (:to code-stream) (:a code-file-v))
	       ))
	 (let ((cpn (pathname code-stream)))
	    (let ((code-type (Pathname-type cpn)))
	       (let ((id-string (out-to-string
				    "File "
				    (:a (Pathname-name cpn))
				    (:q (code-type "." (:a code-type))))))
		  (multi-let (((preamble postamble)
			       (embedded-code-pre-post-strings
				  id-string id-string true)))
		     (out (:a preamble)
			  :%
			  (:a text-file-v)
			  (:a postamble) :%)))))))

(deffunc file-name-stream - (~ Stream) (filespec - Obj)
   (cond ((is Symbol filespec)
	  (let ((e (assq filespec (tv-val litlisp-out-files)))
		- (~ (Lrcd Symbol (~ Pathname) Stream)))
	     (and e (third e))))
	 (t false)))

;;; Used to define "contrived" frags as if they were from source files.
;;; (Typical use: for display of preliminary version of segment before
;;; extracting the final version from the actual source file.)
(defproc define-frag - Void (name - Symbol)
   (discard-rest-of-line "fragment" name)
   (let ((fragment
	    (out-indent *error-output* 3
		(dbg-out txtlisp-dbg*
		    "Recursive call to txtlisp-transduce"
		    " for contrived fragment \""
		    name "\"")
		(let ((frag (with-output-to-string (this-seg)
			       (bind ((building-segment* true))
				  (txtlisp-transduce 
				     *standard-input*
				     this-seg false)))))
		   (dbg-out txtlisp-dbg*
		      "Contrived fragment \""
		      name "\" ends with"
		      :% " [[" (subseq frag
				       (- (string-length frag) 40)
				       (string-length frag)) "]]")
		   frag))))
      (let ((frag-string fragment
			 ;;;;(subst-seg-markers fragment false)
	     ))
	 (with-input-from-string (srm frag-string)
	    (let ((top-frag (make Constructing-frag
			       name ':contrived true
			       (make-string-output-stream)
			       0 !())))
	       (source-stream-extract-fragments
		   srm (list top-frag)
		   "contrived fragment "
		   false)
	       (note-frag top-frag))))))

(deffunc embedded-code-pre-post-strings
                                  - (Multvals preamble postamble - String)
                    (seg-or-frag-name seg-or-frag-ref - (Alt Symbol String)
		     file-seg - Boolean)
   (multi-let (((beg end
		   - (Fun String ((Alt Symbol String)
				  (Alt Symbol String))))
		(seg-def-wrappers
		   txtlisp-control* file-seg)))
      (multi-let (((pre post
		      - (Fun String (String)))
		   (embedded-code-wrappers
		      txtlisp-control*)))
	 (let ((before-string
		  (funcall beg seg-or-frag-name seg-or-frag-ref))
	       (after-string
		  (funcall end seg-or-frag-name seg-or-frag-ref)))
	    (values (funcall pre
			     before-string)
		    (funcall post
			     after-string))))))

;;; This now just provides inputs to (output fns of) 'embedded-code-wrappers'
(defopfunc seg-def-wrappers - (Multvals pre post
					- (Fun String ((Alt Symbol String)
						       (Alt Symbol String))))
			(output-mode - Txtlisp-mode file-seg - Boolean)
			(ignore file-seg)
   (values (\\ (_ segref)
	      (out-to-string
		 :% "<<Define "  segref))
	   (\\ (_ _) (out-to-string ">>"))))

(defopfunc embedded-code-wrappers
                 - (Multvals pre post
			     - (Fun String (String)))
			(output-mode - Txtlisp-mode)
   (values (\\ (begin-string - String) begin-string)
	   (\\ (end-string - String) end-string)))

(defopfunc seg-ref-producer - (Fun String (Symbol))
	                  (output-mode - Txtlisp-mode)
   (\\ (name)
      (let ((n (seg-leading-blanks name)))
	 (out-to-string
	     (:q (n
		  (:q ((> n 0) (:_ n)))
		  "<<Insert: " name ">>")
		 (t "<<Insert: ??" name "??>>"))))))

;;; Returns false if seg is undefined, else the number of leading blanks in
;;; its definition.
(deffunc seg-leading-blanks - (~ Integer) (segname - Symbol)
   (let ((seg (alref (tv-val seg-table) segname)))
      (cond (seg
	     (let ((seg-string (!_defn seg)))
	        (cond (seg-string
		       (repeat :for ((i = 0
					:to (- (string-length seg-string)
					       1)))
			:until (not (char= (string-elt seg-string i) #\Space))
			:result i))
		      (t false))))
	    (t false))))

(defopfunc frag-ref-producer - (Fun String (Symbol Boolean))
	                  (output-mode - Txtlisp-mode)
   (\\ (name defined)
      (out-to-string
	 "<<Insert: " (:q ((not defined) "??"))
		      name
		      (:q ((not defined) "??")) ">>")))

(defproc flag-undefined-name - Void (name - Symbol defined - Boolean
				     srm - Stream)
   (out (:to srm)
	(:q ((not defined) "??"))
	(:a name)
	(:q ((not defined) "??"))))

(defopfunc string-guarder - (Fun String (String)) (m - Txtlisp-mode)
   (\\ (s - String) s))

;;; Hide from Nisp
(defun string-dim (s) (array-dimension s 0))

(defun make-string-array (size)
   (make-array size
	       :element-type 'character
	       :adjustable true
	       :fill-pointer 0))

(specdecl string-dim - (Fun Integer (String))
	  make-string-array - (Fun String (Integer)))

;;; "Guarding" a character or string means "verbatimizing" it so
;;; in a TeX file it will come through as is.

;;; For efficiency, this returns false rather than build a singleton string
;;; that will (almost always) be quickly discarded --
(defopfunc char-guarder - (Fun (~ String) (Char)) (m - Txtlisp-mode)
   (\\ (c - Char) (ignore c) (be (~ String) false)))

(defopfunc char-guard-require-tester - (Fun Boolean (Char))
	                        (m - Txtlisp-mode))

(defproc referring-paper - Void (name - String filename - Sexp &key default)
   (cond (source-file*
	  (on-list (tuple name
			  (make Referring-paper name filename default))
		   referring-papers*))
	 (t
	  (signal-problem referring-paper
	     "Attempt to declare referring paper " name
	     " outside of a source file"
	     :% 5 (:a name) " = " filename))))
;;; -- e.g., (referring-paper "McDermott05"
;;;                           "~/UNIfied/word/pub/Lisp2005/lisp05.txl"
;;;                          :default true)

(defproc ignore-seg - Void (name - Symbol)    ;;;; &rest whatever
   (transduce-to-code-and-text `("ignorable" ,name)
			       true))

;;; For both 'code-seg' and 'file-seg', we assume that the rest of the line
;;; where the call occurs is blank, and it plus its newline are to be discarded.

(specdecl rest-of-line* - String)

(deffunc transduce-to-code-and-text - (Multvals code-version text-version
						- String)
	                            (where - (Lrcd String Symbol)
				     ignore-forms - Boolean)
   (let-fun ()
      (discard-rest-of-line "segment" (second where))
      (cond (building-segment*
             (signal-problem transduce-to-code-and-text
                "Definition of " (:a (car where)) " segment " (cadr where)
                :% " begins in middle of " (:a (car building-segment*))
                " segment " (cadr building-segment*)
                (:proceed "I'll proceed, but expect further problems"))))
      (let ((segment
	       (progn
		  (out-indent *error-output* 3
		     (dbg-out txtlisp-dbg*
			 "Recursive call to txtlisp-transduce"
			 " for " (:a (car where)) " segment \""
			 (cadr where) "\"")
		     (let ((seg (with-output-to-string (this-seg)
				    (bind ((building-segment* where))
				       (txtlisp-transduce 
					  *standard-input*
					  this-seg ignore-forms)))))
			(dbg-out txtlisp-dbg*
			   (:a (car where)) " segment \""
			   (cadr where) "\" ends with"
			   :% " [[" (subseq seg
					    (- (string-length seg) 40)
					    (string-length seg)) "]]")
			seg)))))
;;;;	 (err-out "Ready to subst: " segment :% :%)
	 (values (subst-seg-markers segment true)
		 (subst-seg-markers segment false)))))

(defproc discard-rest-of-line - Void (sort - String name - Sexp)
   ;; Flush whatever comes after (code-seg ...), (file-seg ...), or
   ;; (define-frag ...) on this line --
  (repeat :for ((r = (in :char) :then :again))
     :until (eq r eof*)
     :result (signal-problem code-seg
		"File ends before beginning of code "
		(:a sort) 1 name)
     :until (char= r #\Newline)))

(defproc insert-cxt-into-text - (Mlv Integer Boolean Integer)
	                 (s - String start-seg end-seg - Integer
			  starting-line - Boolean leading-blanks - Integer
			  code-version - Boolean expansion - Stream)
   (cond (code-version
	  (values end-seg starting-line leading-blanks))
	 (t
	  (out (:to expansion)
	      ":."
	      (:a (funcall (string-guarder txtlisp-control*)
			   (string-subseq s start-seg end-seg)))
	      ".:")
	  (values end-seg false 0))))

;;; Debugging hack fast becoming extremely obscure.  Feel free to trash --
(specdecl ch* - Char (outnums* false) - Boolean)

 ;; Rather ad hoc gizmo that works only with ":. ... .:" --
(defproc advance-after-cxt - (Multvals end-pos - Integer
				       line-start - Boolean
				       num-blanks - Integer)
                          (s - String
			   start-cxt end-cxt - Integer
			   starting-line - Boolean leading-blanks - Integer
			   code-version - Boolean expansion - Stream)
   (cond ((not code-version)
	  (values end-cxt false 0))
	 (t
	  (let-fun ()
	     ;; If rest of line is blank, advance to first nonblank on
	     ;; next line, else to first nonblank on this line
	     ;; The cxt stuff inserted must look like blanks in code.
	     (multi-let (((resume-vis start-new-line new-line-pos)
			  (gobble-blank-line end-cxt)))
		(cond (start-new-line
		       (values resume-vis
			       true
			       (- resume-vis new-line-pos)))
		      (starting-line
		       (values resume-vis
			       true
			       (+ leading-blanks (- resume-vis start-cxt))))
		      (t
		       (out (:to expansion)
			    (:_ (- resume-vis start-cxt))
			    (:q (outnums* "1")))
		       (values resume-vis false 0))))
		       
	   :where

  (:def gobble-blank-line - (Multvals resume-after-cxt - Integer
				      start-new-line - Boolean
				      new-line-pos - Integer)
			 (end-cxt - Integer)
;;;;    (trace-around gobble
;;;;       (:> "(gobble: " end-cxt ")")
    (repeat :for ((start-new-line false) - Boolean
		  (after-cxt end-cxt)
		  (slen (string-length s))
		  new-line-pos - Integer)
     :until (= after-cxt slen)
     :within
       (let ((ch (string-elt s after-cxt)))
;;;;	  (!= ch* ch)
	  (:continue
	   :until (not (is-whitespace ch))
	      (cond ((char= ch #\Newline)
		     (cond (start-new-line
			    (err-out "Warning: Multiple blank lines after "
				     :% "... " (:a (string-subseq
						      s start-cxt end-cxt))
					       ".:"
				     :%))
			   (t
			    (!= start-new-line true)
			    (!= new-line-pos (+ after-cxt 1))))))))
       (!= after-cxt (+ *-* 1))
     :result (progn (values after-cxt start-new-line new-line-pos)))
;;;;       (:< (new-pos start-nl nl-pos) "gobble: " new-pos 1 start-nl 1 nl-pos))
    )))))

(defopfunc code-seg-ellipsis - String (m - Txtlisp-mode)
   "...")

(defproc insert-code-seg - (Mlv Integer Boolean Integer)
	                 (s - String start-seg end-seg - Integer
			  starting-line - Boolean leading-blanks - Integer
			  code-version - Boolean expansion - Stream)
   (cond (code-version
	  (let ((new-seg (string-subseq s start-seg end-seg)))
	     (multi-let (((new-starting-line new-leading-blanks)
			  (code-seg-insert new-seg starting-line leading-blanks
					    expansion)))
		(values end-seg new-starting-line new-leading-blanks))))
	 (t
	  (out (:to expansion)
	       (:a (code-seg-ellipsis txtlisp-control*)))
	  (values end-seg false 0))))

(defproc code-seg-insert - (Mlv Boolean Integer)
	                  (seg - String
			   starting-line - Boolean leading-blanks - Integer
			   expansion - Stream)
   (multi-let (((line-start num-blanks)
		(trailing-blanks-after-newline seg)))
;;;;      (err-out id " -- " line-start 1 num-blanks
;;;;	       " / " starting-line
;;;;	       1 leading-blanks)
      (cond (line-start
	     (out (:to expansion)
	       (:q (starting-line
		    (:_ leading-blanks)))
	       (:a (string-subseq
		      seg 0
		      (- (string-length seg)
			 num-blanks)))
	       (:q (outnums* "4")))
	     (values true num-blanks))
	    ((>= num-blanks 0)
	     ;; All blanks
	     (cond (starting-line
		    (values true (- leading-blanks num-blanks)))
		   (t
		    (out (:to expansion)
		       (:_ (- num-blanks))
		       (:q (outnums* "5")))
		    (values false 0))))
	    (t
	     (out (:to expansion)
		(:q (starting-line
		     (:_ leading-blanks)))
		(:a seg)
		(:q (outnums* "6"))
		)
	     (values false 0)))))

(defproc insert-tex-seg - (Mlv Integer Boolean Integer)
	                 (s - String start-seg end-seg - Integer
			  _ - Boolean _ - Integer
			  code-version - Boolean expansion - Stream)
   (cond ((not code-version)
	  (out (:to expansion)
	     (:a (string-subseq s start-seg end-seg)))))
   (values end-seg false 0))

(specdecl (seg-escape-char* #\:) - Char)

(defnisptype Text-handler (Fun (Multvals end-pos - Integer
					 code-starting-line - Boolean
					 code-leading-blanks - Integer)
			       (String Integer Integer Boolean Integer
				       Boolean Stream)))
;; -- args: segment, start-piece, end-piece, 
;; code-starting-line, code-leading-blanks,
;; code-vs-text, out-stream

(defnisptype Litlisp-escape-control
       (Structure end-char - Char
		  seg-out - Text-handler
		  post-seg - (~ Text-handler)))

(specdecl seg-markers* - (Lst (Lrcd Char Litlisp-escape-control)))

(!= seg-markers*
    (list (tuple #\. (make Litlisp-escape-control
			   #\. !'insert-cxt-into-text !'advance-after-cxt))
	  (tuple #\( (make Litlisp-escape-control
			   #\) !'insert-code-seg false))
	  (tuple #\{ (make Litlisp-escape-control
			   #\} !'insert-tex-seg false))))

(specdecl after-cxt* - Integer s* - String)

(deffunc subst-seg-markers - String (s - String code-version - Boolean)
   (let ((slen (string-length s))
	 (delim-len (string-length seg-delimiter*)) - Integer
;;;;	 (output-mode (tv-val output-mode)) - Txtlisp-mode
	 (must-be-guarded (char-guard-require-tester txtlisp-control*))
	 - (Fun Boolean (Char))
	 )
      (!= s* s)
      (with-output-to-string (expansion)
	 (repeat :for ((i = 0) (prev-i -1) - Integer
		       ;; Both of these apply to code output only --
		       ;; True if we haven't seen a nonspace on this line yet --
		       (code-starting-line code-version) - Boolean
		       ;; ... in which case this is the number of spaces seen --
		       (code-leading-blanks 0) - Integer
		       (ch #\Rubout) - Char
		       (seg-ref-produce (seg-ref-producer txtlisp-control*))
		         - (Fun String (Symbol))
		       (string-guard (string-guarder txtlisp-control*))
		         - (Fun String (String))
		       (char-guard (char-guarder txtlisp-control*))
		         - (Fun (~ String) (Char)))
	  :until (>= i slen)
	    (cond ((=< i prev-i)
		   (dbg-save :run-loud s prev-i i
			     ch code-starting-line code-leading-blanks)
		   (signal-problem subst-seg-markers
		      "Infinite loop in subst-seg-markers at " i)))
	    (!= prev-i i)
	    (cond ((and code-version (char= ch #\Newline))
;;;;		   (!= beginning-of-line i)
		   (!= code-starting-line true)
		   (!= code-leading-blanks 0)))
	    (!= ch (elt s i))
	    (cond ((and code-starting-line (char= ch #\Space))
		   (!= code-leading-blanks (+ *-* 1))
		   (!= i (+ i 1)))
		  ((= (mismatch seg-delimiter* s
				:start2 i)
		      delim-len)
		   (copy-seg))
		  ((and (char= ch seg-escape-char*)
			(< i (- slen 2)))
		   (escape-seg))
		  (t
		   (build-seg)))

       :where

 (:def copy-seg - Void ()
    (!= i (+ i delim-len))
    (repeat :for ((j = i :by 1 :to slen)
		  - Fixnum mis - (~ Fixnum))
     :result (progn
	        (dbg-save :run-loud s i)
		(signal-problem subst-seg-markers
		   "Can't find delimiter after position " i " in"
		   :% s))
	(!= mis (mismatch seg-delimiter* s :start2 j))
     :until (or (not mis) (= mis delim-len))
     :result (let ((id (intern
			  (subseq s i j))))
		(!= i (+ j delim-len))
		(let ((e (alref (tv-val seg-table) id))
		      - (~ Prog-segment))
		   (let ((defn (and e (!_defn e)))
			 - (~ String))
		      (cond (code-version
			     (!= < code-starting-line code-leading-blanks >
				 (code-seg-insert
				    defn code-starting-line code-leading-blanks
				    expansion)))
			    (t
			     (out (:to expansion)
;;;;				(:q (defn (:a (leading-blanks defn))))
				(:a (funcall seg-ref-produce id))))))))))

 (:def escape-seg - Void ()
    (let ((next (elt s (+ i 1))))
       (cond ((char= next #\\)
	      (out (:to expansion)
		   (:a (funcall string-guard (string seg-escape-char*)))
		   (:q ((and outnums* code-version) "7")))
	      (!= i (+ i 2)))
	     (t
	      (let ((ctl (alref seg-markers* next false :test #'char=))
		    - (~ Litlisp-escape-control))
		 (cond (ctl
			(repeat :for ((j = i :by 1 :to (- slen 2))
;;;;				      (first-line beginning-of-line)
				      - Integer
				      (ch-j = (elt s i) :then ch-j+1)
				      (ch-j+1 = (elt s (+ i 1))
					      :then (elt s (+ j 2)))
				      (end-char (!_end-char ctl))
				      - Char)
			 :result
			    (progn (dbg-save :run-loud
					     i j s ch-j ch-j+1
					     end-char)
				   (signal-problem subst-seg-markers
				    "Can't find context-ender "
				    "'" (:a end-char seg-escape-char*)
				    "'" " after position " i " in"
				    :% s))
			 :until (and (char= ch-j end-char)
				     (char= ch-j+1 seg-escape-char*))
			 :result
			    (let ()
			       (!= < _ code-starting-line code-leading-blanks >
				   (funcall (!_seg-out ctl) s (+ i 2) j 
					    code-starting-line
					    code-leading-blanks
					    code-version expansion))
			       (cond ((!_post-seg ctl)
				      (!= < i
					    code-starting-line
					    code-leading-blanks >
					  (funcall (!_post-seg ctl)
					     s i (+ j 2)
					     code-starting-line
					     code-leading-blanks
					     code-version expansion))
				      (!= after-cxt* i))
				     (t
				      (!= i (+ j 2)))))))
		       (t
			(out (:to expansion)
			     (:q ((and code-version code-starting-line)
				  (:_ code-leading-blanks)
				  (:e (!= code-starting-line false))))
			     (:a (funcall string-guard
					  (string seg-escape-char*)))
			     (:q ((and outnums* code-version) "8")))
			(!= i (+ i 1)))))))))

 (:def build-seg - Void ()
     (let ((j i) (ch-j ch))
	(repeat
	 :while (funcall must-be-guarded ch-j)
	    (!= ch ch-j)
	    (!= j (+ j 1))
	 :until (= j slen)
	    (!= ch-j (string-elt s j)))
	;; i through j require guarding, but only in
	;; text file
	(cond ((= i j)
	       ;; No guarding required; just output next char
	       (out (:to expansion)
		    (:q (code-version
			 (:q (code-starting-line
			      (:_ code-leading-blanks)))
			 (:a ch)
			 (:q (outnums* "9"))
			 )
			(t
			 ;; Newlines still need special handling --
			 (:e
			   (let ((gch (funcall char-guard ch)))
			      (:o (:to expansion)
				  (:q (gch (:a gch))
				      (t (:a ch)))))))))
	       (cond ((and (char= ch #\Newline)
			   code-version (= i 30))
		     ))
	       (!= i (+ i 1)))
	      (t
	       (out (:to expansion)
		    (:q (code-version
			 (:q (code-starting-line
			      (:_ code-leading-blanks)))
			 (:a (subseq s i j))
			 (:q (outnums* "0"))
			 )
			(t
			 (:a (funcall string-guard (subseq s i j))))))
;;;;	       (cond (code-version
;;;;		      (cond ((char= ch #\Newline)
;;;;			     (!= ch* ch)
;;;;			     (!= ss* s) (!= i* i) (!= j* j)))))
	       ;; Character 'j' remains unhandled; it might be ':', 
	       ;; the escape marker, and it must be seen
	       ;; when you go around the loop again
	       (!= i j)))
	(!= code-starting-line false)))))))

;;; Produce a version that looks right in the current output mode --
(deffunc textify - Void (line - String out-srm - Stream)
;;; This is very similar to 'build-seg' (immediately above); perhaps a 
;;; merge is possible.	The key difference is that 'build-seg' just
;;; extracts a maximal substring requiring guarding, whereas this
;;; procedure must get the entire line out.   (Because 'textify' is
;;; for producing fragments, and 'build-seg' for producing segments.)
;;; Also, this procedure does the tab expansion that 'txtlisp-transduce'
;;; normally does, because 'textify' is called by 'source-file'.
   (let ((string-guard (string-guarder txtlisp-control*))
	   - (Fun String (String))
	 (must-be-guarded (char-guard-require-tester txtlisp-control*)))
     (repeat :for ((i = 0) (slen (string-length line)))
      :while (< i slen)
	(let ((j i) - Integer (ch-j (string-elt line i)) - Char)
	   (repeat
	    :while (funcall must-be-guarded ch-j)
	       (!= j (+ j 1))
	    :until (= j slen)
	       (!= ch-j (string-elt line j)))
	   ;; j is the subscript of the first character that does
	   ;; not require guarding (= slen if all characters from 'i'
	   ;; on require guarding).
	   ;; i through j require guarding
	   (cond ((= i j)
;;;;		  (breakpoint textify "j = " j " ch-j = " ch-j)
		  (out (:to out-srm) (:a ch-j))
		  (!= i (+ i 1)))
		 (t
		  (let ((to-be-guarded 
			   (cond ((find #\Tab line :start i :end j)
				  (with-output-to-string (str)
				     (repeat :for ((pos = i :to (- j 1)))
					(let ((ch (elt line pos)))
					   (cond ((char= ch '#\Tab)
						  (out (:to str) 8))
						 (t (out (:to str)
							 (:a ch))))))))
				 (t
				  (subseq line i j)))))
		     (out (:to out-srm)
			  (:a (funcall string-guard to-be-guarded)))
		     (!= i j))))))))

(deffunc leading-blanks - String (str - String)
   (repeat :for ((i = 0 :to (- (string-length str) 1))
		 - Fixnum
		 :collector whitespace - Char)
    :while (is-whitespace (string-elt str i))
    :collect (string-elt str i)
    :result (coerce whitespace 'string)))

;;; If str ends with newline + k spaces,  returns < t, k>,
;;; else if str is all spaces, returns < nil, length >
;;; else < nil, -1 >.
(deffunc trailing-blanks-after-newline - (Mlv Boolean Integer) (str - String)
    (let ((str-length (string-length str)))
       (repeat :for ((k = (- str-length 1)
			:by -1
			:to 0) - Fixnum
		     ch - Char)
	:result (values false str-length)
	  (!= ch (string-elt str k))
	:while (char= ch #\Space)
	:result (cond ((char= ch #\Newline)
		       (values true (- str-length k)))
		      (t (values false -1))))))

(deffunc load-seg-table - Void (seg-pn - Pathname)
   (let ((seg-table !()))
      (with-open-file (seg-srm seg-pn :direction ':input)
	 (repeat :for ((r = (in (:from seg-srm) :linestring)
			  :then :again)
		       mis segname)
	  :until (eq r eof*)
	  :result nil
	    (!= mis (mismatch "~~" r))
	    (cond ((and mis (= mis 2))
		   (!= segname (intern (subseq r 2 (string-length r))))
		   (repeat :for ((s = (in (:from seg-srm) :linestring)
				    :then :again)
				 - String
				 :collector lines - String)
		    :until (eq s eof*)
		    :result (err-out
			       "Segment-table file ends prematurely")
		    :until (not (mismatch r s))
		    :nconc (list (string #\Newline) s)
		    :result (!= seg-table
				(cons (tuple segname
					     (make Prog-segment
						   segname
						   (cond ((null lines) "")
							 (t
							  (<< string-concat
							      (cdr lines))))
						   false false))
				      *-*)))
		   (!= (tv-val seg-table) seg-table))
		  (t
		   (signal-problem load-seg-table
		      "Uninterpretable segment-table line " r)))))))
		    
;;; Return false if omitting tangling.
(defproc ensure-litlisp-env - Boolean ()
   (ensure-segment-table *standard-input*)
   (tv-val tangling))

(defproc ensure-segment-table - Void (in-stream - Stream)
   (cond ((not (tv-is-bound seg-stream))
	  (let ((seg-pn (merge-pathnames seg-pathname* (pathname in-stream))))
	     (cond ((probe-file seg-pn)
		    (load-seg-table seg-pn)
		    (cond ((not (tv-is-bound seg-table))
			   (!= (tv-val seg-table) !()))))
		   (t
		    (!= (tv-val seg-table)
			!())
		    (out (:to *error-output*)
		       "Segment table " seg-pn
		       :% " doesn't exist -- creating" :%)))
	     (!= (tv-val seg-stream)
		 (note-litlisp-out-file '|segment table| seg-pn))
	     (!= (tv-val changed-segs) !())
	     (after-file-transduction
;;;;                (breakpoint ensure-segment-table
;;;;                   "Segment table post-check")
		(let ((chsgl (tv-val changed-segs))
		      - (Lst Symbol))
		   (!= num-changed-segs* (len chsgl))
	           (cond ((> num-changed-segs* 0)
			  (out (:to *error-output*)
			       "Changed segments: " (reverse chsgl) :%))))
		(let ((segtab
		         (tv-val seg-table))
		      - (Lst (Lrcd Symbol Prog-segment)))
		   (let ((unused (repeat :for ((e :in segtab))
				  :when (not (!_used (second e)))
				  :collect (!_id (second e)))))
		      (cond ((not (null unused))
			     (out (:to *error-output*)
				"Segments never used: " (reverse unused)
				:%))))))))))

(defproc note-litlisp-out-file - Stream (id - Symbol
					 pn0 - (Alt Pathname String Sexp))
   (let ((pn (cond ((consp pn0)
		    (filespec->pathname pn0))
		   (t
		    (->pathname pn0))))
	 - Pathname)
      (cond ((not (tv-is-bound litlisp-out-files))
	     (!= (tv-val litlisp-out-files) !())
	     (after-file-transduction
		(repeat :for ((e :in (tv-val litlisp-out-files))
			      - (Lrcd Symbol (~ Pathname) Stream))
;;;;		   (err-out "Closing " (first e) ": " (second e))
		   (if (second e) (close (third e)))))))
      (cond ((and (not (eq id 'nil))
		  (assq id (tv-val litlisp-out-files)))
	     (signal-problem note-litlisp-out-file
		"code-file id " id " declared more than once")))
      (let ((srm (open pn :direction ':output :if-exists ':supersede)))
	 (!= (tv-val litlisp-out-files)
	     (cons (tuple id pn srm) *-*))
	 srm)))

(defproc fragment-table-to-lux-file - Void ()
   (let ((fragtab (tv-val fragment-table)))
      (note-txtlisp-aux-form '(create-fragment-table))
;;;;        '(!= (tv-val fragment-table) !())
      (repeat :for ((e :in fragtab))
	 (let ((frag (second e)))
	    (cond ((not (!_defined frag))
		   (on-list (!_name frag) (tv-val undefined-frags)))
		  (t
		   (with frag
		      (note-txtlisp-aux-form
			 `(decl ()
			     (on-list (tuple ',!>name
					     (make Prog-fragment
						   ',!>name ',!>source-id false
						   ',!>state
						   ':unused ',!>line-num
						   ',(!_contents frag)))
				      (tv-val fragment-table)))))))))))
