;-*- Mode: Common-lisp; Package: opt; Readtable: ytools; -*-
(in-package :opt)
;;; $Id: basics.lisp,v 1.28 2005/12/02 05:10:14 dvm Exp $

(depends-on %module/ ytools)

;;;;(depends-on (:at :run-time) %ytools/ fileseg)

(depends-on :at-run-time %opt/ types %langutils/ namespace synutils)

;;; Slurp to get macro definitions
(end-header :continue-slurping)

(eval-when (:compile-toplevel :load-toplevel :execute :slurp-toplevel)
   (export '(domain-place-situation find-domain-bdg-val
	     next-generation domain-make-current domain-declares-requirement
             try-requirement-with-name try-domain-with-name)))

(defun add-to-rule-group (x rg)
   (push x (Rule-group-rules rg)))

(defun domain-reset (dom)
   ;; Mark "under construction"
   (setf (Domain-generation dom)
	 -1)
   (setf (domain-get-local-bdgs dom) '())
   (cond ((Domain-inherited-bdgs dom)
          (clrhash (Domain-inherited-bdgs dom)))))

(defun domain-under-construction (dom)
   (< (Domain-generation dom) 0))

(defun domain-mark-current (dom)
   (!= (Domain-generation dom)
       (next-generation)))

; By convention, the actions, etc. of a domain are stored in a
; rule group whose "name" is the domain itself (not the name of the domain)
;;; NO LONGER.  We now just bind it under the name |Own Rule Group| 

(defun own-rule-group (dom)
   (let ((rg (domain-place-rule-group dom '|Own Rule Group|)))
      (cond ((not (is-Rule-group rg))
             (error "Domain's own rule group fumbled ~s (~s)"
                    dom rg)))
      rg))

(defun domain-place-rule-group (dom name)
   (let ((rgb (vartypes-lookup name (domain-get-local-bdgs dom)))
	 rg)
      (cond (rgb
	     (setf rg (Domain-bdg-val rgb)))
	    (t
	     (setf rg (make-Rule-group
			:name name
			:domain dom
			:generation (Domain-generation dom)
			:rules '()))
             (setq rgb (make-Domain-bdg
			  :sym name
			  :type rule-group-type*
			  :val rg
			  :domain dom))
	     (push rgb
		   (domain-get-local-bdgs dom))))
      (cond ((is-Rule-group rg)
             rg)
            (t (format-flg name
                  "Not a rule group (value ~s)"
                  rg)))))

(defun domain-place-situation (dom name)
   (let ((s (place-domain-bdg name dom)))
      (cond ((or (Domain-bdg-unbound s)
		 (and (not (is-Situation (Domain-bdg-val s)))
		      (progn
			  (signal-problem domain-place-situation
			     "Not a situation: (value " s ")"
			     (:proceed "I will rebind to a new situation"))
			  true)))
	     (setf (Domain-bdg-val s) (new-Initial-situation name dom))))
      (Domain-bdg-val s)))

(defun new-Initial-situation (name dom)
   (let ((new
	    (make-Initial-situation
	     :name name
	     :domain nil
	     :frozen true
	     :generation 0  ; initially not up to date
	     :delta '()
	     :sit-index nil)))
      (let ((subdom (new-Domain `(domain ,new))))
;;;;	 (out "Setting namespace of " subdom
;;;;	      " to " (Domain-namespace dom) " [from " dom "]" :%)
;;;;	 (!= (Domain-namespace subdom) (Domain-namespace dom))
	 (setf (Domain-parents subdom) (list dom))
	 (setf (Initial-situation-domain new) subdom)
	 (reset-ancestors subdom)
	 (domain-mark-current subdom))
      new))

(defun Initial-situation-main-parent-domain (isit)
   (let ((subdom (Initial-situation-domain isit)))
      (let ((parents (Domain-parents subdom)))
	 (cond ((and (not (null parents))
		     (matchq (domain ?_)
			     (Domain-name subdom)))
		(car parents))
	       (t
		(signal-problem Initial-situation-main-parent-domain
		   :fatal
		   "Can't find main parent of " isit))))))

(defun try-requirement-with-name (name create)
   (multiple-value-bind (v okay ty)
                        (get-global-opt-sym name)
      (cond (okay
	     (cond ((eq ty requirement-type*) v)
		   (t false)))
            (create
             (let ((new (make-Requirement
                           :name name :builtins '() :implies '())))
                (set-global-opt-sym name new requirement-type*)
                new))
            (t false))))

(defun must-domain-with-name (name)
   (let ((dom (try-domain-with-name name nil)))
      (cond ((not dom)
             (error "Undefined domain ~s" name)))
      dom))
                    
(def-op load-domain-with-name (name)
   (or (try-domain-with-name name false)
       (signal-problem load-domain-with-name
	  "Can't load domain given only its name: " name)))

(defun try-domain-with-name (name create)
   (multiple-value-bind (v okay ty)
			(get-global-opt-sym name)
      (cond (okay
	     (cond ((eq ty domain-type*) v)
		   (t false)))
            (create
	     (let ((new (new-Domain name)))
	        (set-global-opt-sym name new domain-type*)
		new))
            (t nil))))

;;; 'uri' is interned
(defun uri-place-opt-name (uri &optional (sym (uri-extract-sym uri)))
   (or (getf (uri-plist uri) 'opt-name)
       (let ((newname (new-global-name sym)))
	  (!= (get newname 'uri) uri)
	  (!= (getf (uri-plist uri) 'opt-name)
	      newname)
	  newname)))

#+allegro
(defun uri-extract-sym (uri)
   (let ((p (net.uri:uri-parsed-path uri)))
      (cond ((member p '(nil (:absolute) (:absolute ""))
		     :test #'equal)
	     'thing)
	    (t
	     (let ((pn (Pathname-name (->pathname (lastelt p)))))
	        (or pn 'thing))))))

#+allegro
;;; Produce the shortest unambiguous URI that captures the information in 'uri'.
;;; intern if 'do-intern'; check that URI actually says "Opt" if 'must-be-opt'.
(defun uri-opt-name (uri must-be-opt do-intern)
   (let ((q (uri-query-parse (net.uri:uri-query uri)))
	 (path (->pathname (net.uri:uri-path uri))))
      (let ((path-ont-name (Pathname-name path))
	    (path-notation-name (Pathname-type path))
	    (q-ont-name (alref q 'ontol))
	    (q-notation-name (alref q 'notation)))
	 (let ((ont-str (and q-ont-name
			     (cond ((or (not path-ont-name)
					(not (string= path-ont-name
						      q-ont-name)))
				    (out-to-string
				       "ontol=" (:a q-ont-name)))
				   (t false))))
	       (notation-str (and q-notation-name
				  (cond ((or (not path-notation-name)
					     (not (string= path-notation-name
							   q-notation-name)))
					 (out-to-string
					    "notation=" (:a q-notation-name)))
					(t false)))))
	    (cond ((and must-be-opt
			(not (string= notation-str "opt")))
		   (note-defective-exp
		      ((uri) "URI specifies notation"
			     " other than Opt: "
			     uri)
		      :target uri
		      :fatal)))
	    (let ((uri1
		     (make-instance 'net.uri:uri
			   :scheme (net.uri:uri-scheme uri)
			   :host (net.uri:uri-host uri)
			   :path (net.uri:uri-path uri)
			   :query
			       (cond ((and ont-str notation-str)
				      (out-to-string (:a ont-str)
						     "&" (:a notation-str)))
				     (t (or ont-str notation-str))))))
	       (let ((uri2
			(cond (do-intern
			       (net.uri:intern-uri uri1 opt-uri-space*))
			      (t uri1))))
		  (!= (getf (net.uri:uri-plist uri2)
			    'notation)
		      notation-str)
		  uri2))))))

;;; Parse prop1=val1&prop2=val2&...&propK=valK into alist 
;;;   ((prop1 val1) ...), 
;;; where each propI is a symbol in opt package, and each valI is
;;; a string.  Unescaping is performed.
(defun uri-query-parse (q)
   (declare (type string q))
   (cond (q
	  (let ((q-len (string-length q)))
	     (let-fun ()
		(repeat :for ((qmark-pos = (or (position #\? q) q-len)
				 :then (or (position #\? q :start (+ prev 1))
					   q-len))
			      (prev = 0 :then (+ qmark-pos 1))
			      :collector alist-entries)
		 :collect
		   (let ((equal-pos (position #\= q :start prev :end qmark-pos)))
		      (cond (equal-pos
			     (tuple (intern (unescape (subseq q prev equal-pos))
					    :opt)
				    (unescape (subseq q (+ equal-pos 1) qmark-pos))))
			    (t
			     (tuple (intern (unescape (subseq q prev qmark-pos)))
				    ""))))
		 :until (= qmark-pos q-len))

	      :where

		(:def unescape (str)
		    (declare (type string str))
		    (let ((str-len (length str))
			  (first-percent (position #\% str)))
		       (cond (first-percent
			      (repeat :for ((percent-pos = first-percent
						   :then (position #\% str :start (+ prev 3)))
					    (prev = 0 :then (+ percent-pos 3))
					    (res ""))
			       :while percent-pos
				 (!= res (concatenate
					    'string
					    *-*
					    (subseq str prev percent-pos)
					    (escape-seq->char (subseq str
								      (+ percent-pos 1)
								      (+ percent-pos 3)))))
			       :result (concatenate
					  'string
					  res
					  (subseq str prev str-len))))
			     (t str))))

		(:def escape-seq->char (estr)
		   (code-char
		      (bind ((*read-base* 16))
			 (read-from-string estr)))))))
	 (t !())))

#+allegro
(defun uri->string (uri)
   (with-output-to-string (ssrm) (net.uri:render-uri uri ssrm)))

;;; This is a special package for global names, to make sure that we can
;;; always make up a new one
(defvar gbl-package* (make-package ':globals :nicknames '(:gb)))

(defvar global-name-count* 0)

(defun new-global-name (name)
      (repeat
         (!= global-name-count* (+ *-* 1))
       :within (multi-let (((s status)
			    (intern (out (:to :string)
				         (:a name) "-" global-name-count*)
				    gbl-package*)))
		  (:continue
		   :while status
		      (unintern s gbl-package*)
		   :result
		      (progn
			 (export s gbl-package*)
			 s)))))

;;; It counts if the domain is the one *defining* the requirement.
(defun domain-declares-requirement (dom reqname)
  ;;;;(domain-make-current dom) <- causes infinite recursion through
  ;;;;                              domain-for-causer-cache in optop
  (let ((req (try-requirement-with-name reqname false)))
     (cond ((not req)
	    (signal-problem domain-declares-requirement
	       "Undefined requirement " reqname))
	   (t
	    (let ((req-builtin-domain
		     (cond ((= (len (Requirement-builtins req)) 1)
			    (car (Requirement-builtins req)))
			   (t false))))
	       (or (and req-builtin-domain
			(exists (anc :in (Domain-ancestors dom))
			   (same-domain anc req-builtin-domain)))
		   (exists (anc :in (Domain-ancestors dom))
		      (exists (r :in (Domain-requirements anc))
			 (eq (Requirement-name r) reqname)))))))))

(defun requirement-implications (req)
   (labels ((pursue (req)
               (adjoin req
                       (mapcan #'pursue
                               (Requirement-implies req))
                       :test #'eq)))
      (pursue req)))

(defun symbol-any-value (sym dom)
   (let ((bdg (place-domain-bdg sym dom)))
      (cond ((Domain-bdg-unbound bdg)
             (multiple-value-let (v okay ty)
				 (get-global-opt-sym sym)
				 (ignore ty)	       
                (cond (okay `(global ,v))
                      (t nil))))
            (t
             `(local ,(Domain-bdg-val bdg))))))

(defun find-domain-bdg-val (sym dom)
   (let ((bdg (find-domain-bdg sym dom)))
      (cond (bdg (Domain-bdg-val bdg))
            (t nil))))

; When place-domain-bdg finds a binding, it copies it to the local table.
; Side effects to local bdg are seen in original scope.
; The following makes a new binding, which does not share with the old.
; This is apparently unused.
(defun domain-rebind (sym val dom)
   (let ((inh (place-domain-inherited-bdgs dom)))
      (setf (gethash sym inh)
            (make-Domain-bdg
                :sym sym
                :val val
                :domain dom))))

(defun place-local-domain-bdg (sym dom)
   (let ((pb (vartypes-lookup sym (domain-get-local-bdgs dom))))
      (cond ((not pb)
	     (setq pb (make-Domain-bdg
				   :sym sym
				   :type univ-type*
				   :val unbound-sym-marker*
				   :domain dom))
	     (push pb (domain-get-local-bdgs dom))))
      pb))

(def-op domain-make-current (dom))

;;; Check that ancestors haven't changed.
;;; Returns true if it had to do something, i.e., if the dom
;;; was not current before this method was called.
(def-meth domain-make-current ((dom Domain))
   (cond ((domain-is-current dom)
	  false)
	 (t
	  (reset-ancestors dom)
	  (domain-mark-current dom)
	  false)))

(defun domain-is-current (dom)
   (and (not (null (Domain-ancestors dom)))
        (let ((g (Domain-generation dom)))
           (dolist (a (Domain-ancestors dom) t)
              (cond ((> (Domain-generation a) g)
                     (return nil)))))))

; This really should check if dom inherits symbols
; multiply.
; Also: need to formalize order to look for domain vars in

(defvar universal-ancestors* '()
   "Domains inherited by absolutely every domain except themselves")

(defvar domain-current-dbg* false)

(defun reset-ancestors (dom)
   (let ((ancl (remove dom universal-ancestors*)))
      (dbg-out domain-current-dbg*
	 dom " out of date -- refreshing" :%)
;;;;      (cond ((memq dom ancl)
;;;;	     (signal-problem reset-ancestors
;;;;		"Resetting ancestors of universal ancestor " dom
;;;;		:% "Restarting opt is probably the only way to recover")))
      ;; This is important, but must be done by caller of 'reset-ancestors',
      ;; usually after further initialization of 'dom'.
      ;;;;(setf (Domain-generation dom) 0)
      (cond ((Domain-inherited-bdgs dom)
             (clrhash (Domain-inherited-bdgs dom))))
      (labels ((note (d)
		  (domain-make-current d)
		  ;(format t "Noting ~s~%" d)
		  ;(setq ancl (adjoin d ancl :test #'eq))
		  (setq ancl
			(nconc
			   (mapcan #'(lambda (a)
					(cond ((some #'(lambda (r)
							 (same-domain
							    a r))
						     ancl)
					       '())
					      (t (list a))))
				   (Domain-ancestors d))
			   ancl))))
	 (domain-walk-immed-ancestors #'note dom)
         ; Make sure domain is in its own ancestor list
	 (setf (Domain-ancestors dom)
               (cons dom (remove dom ancl :test #'same-domain)))
;;; This must now be done by caller of reset-ancestors.
;;;;	 (setf (Domain-generation dom)
;;;;	       (next-generation))
	 ancl)))

;;; "Immediate ancestors" are parents and builtins from requirements.
(defun domain-walk-immed-ancestors (fn dom)
	 (dolist (par (Domain-parents dom))
            (funcall fn par))
         (labels ((collect-req (req)
		     (dolist (b (Requirement-builtins req))
			(funcall fn b))
                     (dolist (im (Requirement-implies req))
                        (collect-req im))))
	    (dolist (req (Domain-requirements dom))
               (collect-req req))))

(defun new-Domain (name)
   (multi-let (((name uri)
		(cond ((uri-p name)
		       (values (intern (uri-fragment name))
			       name))
		      (t
		       (values name false)))))
      (let ((new (make-Domain
		    :name name
		    :uri uri
		    :generation (next-generation)
		    :ancestors '()
		    :parents '()
		    :requirements '()
		    :local-bdgs (make-Ref '()))))
	 new)))

(defvar generation* 0
   "Counter used to mark 'freshness' of domain, which should be fresher than all its ancestors")

(defun next-generation ()
   (setq generation* (+ generation* 1)))

; (defun const-of-type (c pt vartypes)
;    (let ((tester (get-is-tester pt true vartypes false)))
;       (and tester
; 	   (not (eq tester '*untestable))
; 	   (funcall tester c))))

;;;       (cond ((fluent-type-p pt)
;;;	      (const-of-type c (fluent-type-base pt)))
;;;	     ((expression-type-p pt)
;;;	      (const-of-type c (expression-type-actual pt)))
;;;	     (t nil))))

(defun arg-subtypes (at1 at2)
   (cond ((null at1)
          (cond ((null at2) t)
                (t nil)))
         ((null at2)
          nil)
         ((eq (car at1) '&rest)
          (cond ((eq (car at2) '&rest)
                 (arg-subtypes (cdr at1) (cdr at2)))
                (t nil)))
         ((eq (car at2) '&rest)
          nil)
         ((is-subtype (car at1) (car at2))
          (arg-subtypes (cdr at1) (cdr at2)))
         (t nil)))

(defun pathname-new-extension (pname ext)
   (make-pathname
       :host (pathname-host pname)
       :device (pathname-device pname)
       :directory (pathname-directory pname)
       :name (pathname-name pname)
       :type ext))

(defun flag-wrong-type (ob name type)
   (format-flg name
      "Wrong type (value ~s, not of type ~t)"
      ob type))

;(defvar universal-ancestors*)

; Inherited by everyone:

(defvar basic-domain*
        (let ((bd (make-Domain
                     :name 'opt-basics
                     :generation 0
                     :ancestors '()
                     :parents '()
                     :requirements '()
                     :local-bdgs (make-Ref (list opt-type-sys*))
                     :inherited-bdgs nil)))
           (setf (Domain-ancestors bd) (list bd))
           (set-global-opt-sym 'basic bd domain-type*)
           bd)
  "Domain inherited by all other domains")

(def-file-segment univ-ancs ()
   (setq universal-ancestors* (list basic-domain*))
)

(set-global-opt-sym 'opt-basics basic-domain* domain-type*)

(defvar pddl2.1-compatible* false)
(defvar pddl-type-synonyms-defined* false)

(defun switch-pddl-mode ()
   (cond (pddl2.1-compatible*
	  (!= *print-case* ':upcase) ; -- no reason, just tradition
	  (!= pddl2.1-compatible* false))
	 (t
	  (!= *print-case* ':downcase)
	  (!= pddl2.1-compatible* true)
	  (cond ((and (not pddl-type-synonyms-defined*)
		      (not (eq 'Either 'either)))
		 ;; We're in case-sensitive mode, but PDDL isn't,
		 ;; so we have to define a bunch of synonyms
		 (syn-type-former either Alt (:system opt))
		 (repeat :for ((ty-name :in '(number float rational integer ratio char
					      symbol string))
			       (opt-package (find-package :opt)))
		    (!= (global-declaration ty-name opt-type-sys*)
			(global-declaration
			   (intern (string-capitalize (Symbol-name ty-name))
				   opt-package)
			   universal-type-sys*
			   true)))
		 (!= pddl-type-synonyms-defined* true)))))
   pddl2.1-compatible*)


(defun domain-type (d dom)
;;;;   (cond (+namespace-mode+
;;;;	  (!= d (expression-symbols-place d dom))))
   (cond ((and pddl2.1-compatible*
	       (is-Symbol d)
	       (not (var-lookup d (domain-place-env dom))))
	  ;; To fit with competition rules, we have to declare d even
	  ;; if undeclared
	  (create-local-bdg
	     d type-type* 
	     (new-Type
		nisptype::boring-class* d '() univ-type* '()
		(list (tuple 'nisptype::home dom)))
	     dom)))
   (designated-type d false
		    (make-Env
		       true
		       (cond ((eq dom basic-domain*) (list basic-domain*))
			     (t (cons dom (list basic-domain*)))))))

#| This stuff is redundant; duplicates exp-env-symbols-place from desig.lisp.
   Of course, we don't need either one.

(defun expression-symbols-place (e dom)
   (declare (ignorable dom))
   (cond (+namespace-mode+
	  (let ((env (domain-place-env dom))
		(ns (Domain-namespace dom)))
	     (exp-namespace-symbols-place e ns env)))
	 (t e)))
|#

(defun typesee-ns (td)
   (nity::ensure-debug-env-set)
   (typesee (designated-type (desig-syms-resolve td nity::debug-env*)
			     true nity::debug-env*)))

;;; Does a "little" bit of type-designator symbol resolution.
;;; This is a kludge, which should be used *>* only for deducing
;;; coercion patterns.
(defun desig-syms-resolve (e env)
   (declare (ignorable env))
   (let-fun ()
      (cond ((-- +namespace-mode+)
	     (let ((ns (env-find-namespace env)))
		(cond (ns
		       (let-fun ((:def rebuild (sub-e)
				      (cond ((is-Qvar sub-e)
					     sub-e)
					    ((is-symbolish sub-e)
					     (cond ((is-Symbol sub-e)
						    (symbol-namespace-or-env-place
					               sub-e ns env))
						   (t sub-e)))
					    ((atom sub-e) sub-e)
					    ((eq (car sub-e) ':!sym)
					     (cadr sub-e))
					    (t
					     (<# rebuild sub-e)))))
			    (rebuild e)))
		      (t
		       (purge-\:!sym e)))))
	    (t
	     (purge-\:!sym e)))
    :where
       (:def purge-\:!sym (sub-e)
	  (cond ((or (atom sub-e) (is-Qvar sub-e))
		 sub-e)
		((eq (car sub-e) ':!sym)
		 (cadr sub-e))
		(t
		 (<# purge-\:!sym sub-e))))))

;;; Even kludgier
(defun dsr (x dom)
   (desig-syms-resolve x (domain-place-env dom)))

;;; Returns < canonized-symbol, flagged-symbol >.
;;; If 'local' = true and symbol comes from a different namespace,
;;; 'canonized-symbol' = false, and 'flagged-symbol' describes the problem.
;;; If 'local' = false, whatever symbol you find is okay.
(defun symbol-resolve (sym dom local)
   (cond ((-- +namespace-mode+)
	  (let ((ns (Domain-namespace dom))
		(env (domain-place-env dom)))
	     (multi-let (((newsym found-as-local)
			  (symbol-namespace-or-env-place sym ns env)))
		(cond ((or (not local)
			   found-as-local)
;;;;			   (not (is-Canonized-symbol newsym))
;;;;			   (eq (Canonized-symbol-space newsym)
;;;;			       ns))
		       (values newsym newsym))
		      (t
;;;;		       (dbg-save sym dom newsym ns env)
;;;;		       (breakpoint symbol-resolve
;;;;			  "Name " newsym " should be in namespace " ns
;;;;			  :% " instead in "
;;;;			  (:q ((is-Canonized-symbol newsym)
;;;;			       (Canonized-symbol-space newsym))
;;;;			      (t "global env")))
		       (values
			  false
			  (flagexp
			     (out-to-string "Name to be bound in domain "
					    dom
					    " comes from"
					    (:q ((is-Canonized-symbol newsym)
						 " nonlocal namespace "
						 (Canonized-symbol-space newsym))
						(t " global environment"))
					    ":")
			     newsym)))))))
	 (t
	  (values sym sym))))

;;;;(defun symbol-place (sym dom)
;;;;   (symbol-namespace-or-env-place
;;;;       sym (Domain-namespace dom) (domain-place-env dom)))

(defun is-type-name (sym dom)
   (let ((b (find-domain-bdg sym dom)))
      (and b (eq (Vartype-type b) type-type*))))

(defmacro set-fields (name-pref ob &rest fields)
   (let ((obvar (gensym)))
      (let ((setters '()))
         (cond ((= (len fields) 1)
                (!= fields (car fields))))
	 (repeat :for ((fl fields (cddr fl)))
	  :until (null fl)
	    (!= setters
		`((!= (,(cond ((is-Symbol (car fl))
			       (build-symbol (< name-pref) - (< (car fl))))
			      (t
			       (signal-problem set-fields
				  "Bogus field: " (car fl))))
		       ,obvar)
		      ,(cadr fl))
		  ,@*-*))
	  :result (!= setters (reverse *-*)))
	 `(let ((,obvar ,ob))
	     ,@setters
	     ,obvar))))

(defmacro out-defective-exp (&rest out-stuff)
   `(make-Defective-exp
       :observation (\\ (target srm) (ignore target)
		       (out (:to srm) ,@out-stuff))))

(defun flag-if-ill-formed (e ill-formed-subexps)
   (cond ((null ill-formed-subexps)
	  e)
	 ((is-Flagged-subexpression e)
	  (!= (Flagged-subexpression-ill-exps e)
	      (append *-* ill-formed-subexps))
	  e)
	 (t
	  (new-Flagged-subexpression e ill-formed-subexps))))

(defun format-flg (e &rest format-args)
   (new-Flagged-subexpression
      e (list (make-Defective-exp
		 ;;;; :has-target false
		 :target false
		 :observation
		 (\\ (_ srm)
		    (<< format srm format-args))))))

(defun synerrs-flag (synerrs target list-already)
   (cond ((null synerrs)
	  (cond (list-already target)
		(t `(,target))))
	 (t
	  (list (new-Flagged-subexpression
		   target
		   (synerrs-defective-exps synerrs target))))))

(defun synerrs-defective-exps (synerrs target)
   (cond ((null synerrs) '())
	 (t
	  (list (make-Defective-exp
		   :target target
		   :observation
		      (\\ (targ srm)
			 (out (:to srm)
			    "Syntax errors: " synerrs
			    (:pp-nl :linear)
			    " in " targ :%))
		   :signaler
		      (\\ (this-exp)
			 (signal-problem synerrs-flag
			    "Defect: " this-exp
			    (:continue "I'll overlook them"))))))))

(defmacro def-deductive-macro (name domain args &rest body)
   `(define-ded-macro
       ',name
       (must-domain-with-name ',domain)
       #'(lambda ,args ,@body)))
  
(defun define-ded-macro (name dom fcn)
   (cond (+namespace-mode+
	  (let ((ns (Domain-namespace dom)))
	     (cond (ns
		    (let ((cs (sym-ns-resolve name ns false)))
		       (cond (cs (!= name cs)))))
		   (t
		    (signal-problem define-ded-macro
		       "Domain " dom " has no namespace"))))))
   (let ((fctrb (var-place-and-lookup name (domain-place-env dom))))
     (cond ((and fctrb (is-Predicate (Domain-bdg-val fctrb)))
            (setf (Domain-generation dom)
                  (next-generation))
            (setf (Predicate-macro (Domain-bdg-val fctrb))
                  fcn))
           (t
            (error "Can't define '~s' as deductive macro ~%   ~
                    (must be declared as predicate first)"
                   name)))))

(defun find-domain-in-vartypes (env)
   (repeat :for ((x :in (nity::Env-vartypes env)))
    :result false
    :until (is-Domain x)
    :result x))
