;-*- Mode: Common-lisp; Package: opt; Readtable: ytools; -*-
(in-package :opt)
;;; $Id: checker.lisp,v 1.17 2005/08/27 05:29:58 dvm Exp $

;;; Main entry points for file syntax checking

(depends-on %module/ ytools)

(depends-on :at-run-time %opt/ definers parsers)

(eval-when (:compile-toplevel :load-toplevel :execute :slurp-toplevel)
   (export '(define when-to-dump* opt-load last-entity-defined*)))

(declaim (special flag-count*))  ; defined in typecheck

(defvar when-to-dump* ':always
   "When to write out syntax-checked expressions: always, never, or when-flagged")


(defvar current-opt-load-specs* false)

(defmacro opt-load (&rest filespecs)
   `(do-opt-load ',filespecs))

(defun do-opt-load (filespecs)
   (cond ((and (null filespecs)
	       current-opt-load-specs*)
	  (!= filespecs current-opt-load-specs*))
	 (t
	  (!= current-opt-load-specs* filespecs)))
   (let ((pnl (filespecs->pathnames filespecs)))
      (repeat :for ((pn :in pnl))
         (let ((out-pn (pathname-find-associate
			  pn 'checked '"chk" false)))
	    (cond (out-pn
		   (out (:to *query-io*)
		      (:pp-block
		         (:pp-ind :block 3)
			 "Sending checked output to "
			 1 (:pp-nl :fill) out-pn :%))
		   (let ((res (opt-file-syncheck pn out-pn)))
		      (out "Result for " (:a pn) ": "
			   :% 5 res :%)))
		  (t
		   (out "Can't figure out where .chk file goes for "
			pn :%)))))))

(defvar opt-file-extensions* '("opt" "pddl" nil))

(defun opt-file-syncheck (in-file &optional (out-file nil))
  (!= in-file (->pathname *-*))
  (let (pn)
     (repeat :for ((ext :in opt-file-extensions*))
      :result (signal-problem opt-file-syncheck
		 "No input file found with pathname "
		 in-file
		 :% " . " opt-file-extensions*)
	(!= pn (merge-pathnames in-file
				(make-pathname :type ext)))
      :until (probe-file pn)
      :result (!= in-file pn))
;;;;     (out "pn = " pn " in-file = " in-file :%)
     (with-open-file (insrm in-file :direction ':input)
		   (cond ((and (not out-file)
			       (not (eq when-to-dump* ':never)))
			  (setq out-file
				(pathname-new-extension in-file "chk"))))
		   (maybe-bind-uri 
		      web-mode*
		      (\\ ()
			 (let ((pn-dir (Pathname-directory pn)))
			    (cond ((or (not pn-dir)
				       (eq (first pn-dir) ':relative))
				   (!= pn
				       (merge-pathnames
					  *-* *default-pathname-defaults*))))
			    (pathname-to-uri (pathname-uri-bug-workaround pn))))
		      (\\ ()
			 (out-file-insrm-syncheck out-file insrm))))))

(defun pathname-uri-bug-workaround (pn)
   (let-fun ()
      (merge-pathnames
	 (make-Pathname
	    :directory (<# space-subst (Pathname-directory pn)))
	 pn)
    :where
       (:def space-subst (str)
	   (cond ((is-String str)
		  (coerce
		     (<! (\\ (c)
			    (cond ((char= c #\space)
				   (list #\% #\2 #\0))
				  (t (list c))))
			 (coerce str 'list))
		     'string))
		 (t str)))))

;;;;(def-meth load-domain-with-name ((sym symbol))
;;;;   (sym-load-domain sym (symbol-name sym)))
;;;;
;;;;(def-meth load-domain-with-name ((name string))
;;;;   (sym-load-domain (intern name) name))

(def-meth load-domain-with-name ((name uri))
   (let ((dom-name (getf (uri-plist name) 'opt-name)))
      (cond ((not dom-name)
	     (signal-problem load-domain-with-name
		"uri not associated with a domain name: " name)))
      (cond ((getf (uri-plist name) 'loaded)
	     (or (try-domain-with-name dom-name false)
		 (signal-problem load-domain-with-name
		    "Domain " dom-name " is not loaded although this uri"
		    " is flagged as loaded:" :% name)))
	    (t
	     (opt-uri-syncheck name false)
	     (prog1 (or (try-domain-with-name dom-name false)
			(signal-problem load-domain-with-name
			   "Loading " name
			   :% " failed to load domain with name " dom-name))
	            (!= (getf (uri-plist name) 'loaded)
			true))))))

;;; No longer used:
(defun sym-load-domain (sym sym-name)
   (cond (web-mode*
	  (let ((container-uri
		   (copy-uri (parse-uri sym-name)
			     :fragment false)))
	     (out (:to *query-io*)
		"Loading " container-uri :%)
	     (opt-uri-syncheck container-uri false)
	     (or (try-domain-with-name sym false)
		 (progn
		    (signal-problem sym-load-domain
		       "Loading " container-uri
		       :% " failed to define domain " (:a sym-name)
		       (:proceed "I will proceed with domain undefined"))
		    false))))
	 (t (try-domain-with-name sym false))))

(defun opt-uri-syncheck (uri out-file)
   (multi-let ((uri
		(cond ((or (is-String uri)
			   (uri-p uri))
		       (intern-uri uri))
		      (t
		       (signal-problem opt-uri-syncheck
			  "Illegal uri: " uri)))))
      (case  (uri-scheme uri)
	 (:file
	    (maybe-bind-uri
	       true
	       (\\ () (intern-uri uri opt-uri-space*))
	       (\\ () 
		  (opt-file-syncheck (uri-path uri) out-file))))
	 (:http
	    (multi-let (((text code _ trans-uri)
			 (handler-case (do-http-request uri)
			    (error (e)
	     ;;;;			 (dbg-save e)
			       (note-defective-exp
				  ((uri) "While trying to read " uri
					 :% "  encountered "
					 (:a (condition-display-string e)))
				  :target uri)))))
	       (cond ((= code 200)
		      (maybe-bind-uri
			 true
			 (\\ () (intern-uri trans-uri opt-uri-space*))
			 (\\ () 
			    (with-input-from-string (insrm text)
			       (out-file-insrm-syncheck out-file insrm)))))
		     (t
		      (note-defective-exp
			 ((uri) "Unsuccessful HTTP-GET: " uri)
			 :target uri
			 "Can't find uri: " uri
			 :% "Return code: " code " Actual uri: " trans-uri
			 :% "Return text: " :% text :%)))))
	 (t
	  (signal-problem opt-uri-syncheck
	     "Can't read domain from uri " uri)))))

(defun maybe-bind-uri (condition make-uri then)
   (cond (condition
	  (let ((interned-uri (funcall make-uri)))
	     (cond ((memq interned-uri uri-loading-stack*)
		    (signal-problem maybe-bind-uri
		       "Attempt to load uri " interned-uri
		       :% " which is already in the process of being loaded"))
		   (t
		    (bind ((uri-loading-stack*
			      (cons interned-uri uri-loading-stack*)))
		       (funcall then))))))
	 (t
	  (funcall then))))

(defun out-file-insrm-syncheck (out-file insrm)
   (cond (out-file
	  (with-open-file (outsrm out-file
			     :direction ':output
			     :if-exists ':supersede
			     :if-does-not-exist ':create)
;;;;	    (out "Entering out-file-insrm-syncheck, parents = "
;;;;		 (langutils::Namespace-parents requirement-namespace*) :%)
	     (syncheck-and-dump insrm outsrm)))
	 (t
	  (syncheck-and-dump
	      insrm
	      (and (not (eq 'when-to-dump* ':never))
		   *error-output*)))))

(declaim (special nisptype::post-file-transduce-hooks*
		  ytools::lisp-preferred-case*))

(defvar pddl-case-readtable*
        (let ((rt (copy-readtable ytools-readtable*)))
	   (setf (readtable-case rt)
	         (case ytools::lisp-preferred-case*
		    ((:lower) ':downcase)
		    (t ':upcase)))
	   rt))

(defun syncheck-and-dump (insrm outsrm)
   (with-post-file-transduction-hooks
      (bind ((stream-being-synchecked* insrm))
	 (let ((total-flag-count 0))
	    (let ((r nil))
	       (bind ((*package* *package*)
		      (*readtable*
			 (cond (pddl2.1-compatible*
				pddl-case-readtable*)
			       (t ytools-readtable*))))
		  (bind ((*print-case*
		            (readtable-case *readtable*)))
		     (loop
			(setq r (read insrm nil 'end-of-file))
			(cond ((member r '(end-of-file (stop-syncheck))
				       :test #'equal)
			       (return)))
			(multiple-value-let (n _)
					    (expand-and-dump r outsrm)
			   (cond (outsrm (out (:to outsrm) :%)))
			   (setq total-flag-count
				 (+ total-flag-count n)))))))
	    `(flagged ,total-flag-count expressions)))))

; First val is number of flagged subexpressions; second is t if form was
; nontrivial.
(defun expand-and-dump (r out-file)
   (bind ((flag-count* 0))
      (let ((nontrivial nil))
	 (multi-let (((mh _)
		      (opt-form-handler r global-opt-env* false)))
	    (cond ((consp mh)
		   (case (car mh)
		      (macro
		       (multiple-value-bind (stuff nonsense)
					    (funcall (cadr mh) r)
					    (declare (ignore nonsense))
			  (dolist (e stuff)
			     (multiple-value-bind (n nontriv)
						  (expand-and-dump e out-file)
				(setq flag-count*
				      (+ flag-count* n))
				(if nontriv (setq nontrivial t))))))
		      (top-level
		       (setq nontrivial t)
		       (multi-let (((flagged-form namespace)
				    (funcall (cadr mh) r)))
;;;;			  (!= save-flagged* flagged-form)
;;;;			  (cond ((is-Flagged-subexpression flagged-form)
;;;;				 (dbg-save flagged-form)
;;;;				 (breakpoint syncheck-and-dump
;;;;				    "Top-level flagged subexpression: "
;;;;				    flagged-form)))
 
			  (bind ((namespace* namespace))
			     (maybe-dump flagged-form out-file))))))
		  (t
		   (maybe-dump r out-file)))
	    (values flag-count* nontrivial)))))
               
;;;;(defvar last-out*)

(defun maybe-dump (e outsrm)
;;;;   (!= last-out* e)
   (cond ((and outsrm
	       (or (eq when-to-dump* ':always)
		   (and (eq when-to-dump* ':when-flagged)
			(> flag-count* 0))))
	  (form-dump e outsrm))))

(defun form-dump (e outsrm)
          (let ((*print-pretty* t)
	        (*print-level* nil)
	        (*print-length* nil))
            (print e outsrm)))

(def-opt-form-handler in-type-system top-level (in-type-form)
   (let ((sysname (cadr in-type-form)))
      (cond ((find-type-system sysname false)
	     (nisptype::switch-into-type-system sysname)
	     (values in-type-form false))
	    (t
	     (values
		`(in-type-system ,(flagexp "Not a type system" sysname))
		false)))))

(defvar last-entity-defined* false)

;;; The expectation is that checking the syntax of a domain will add
;;; an entry for that domain to the opt-domain-table*.  Later forms can 
;;; be checked as if they had been included in the original spec.

(def-opt-form-handler define top-level (def-form)
   (bind ((aux-defs* ':not-allowed)
	  (eval-aux-defs* true)
	  (auxfn-counter* 0))
      (!= last-entity-defined* false)
      (cond ((and (consp (cadr def-form))
		  (symbolp (caadr def-form)))
	     (multi-let (((h _)
			  (opt-form-handler (cadr def-form) global-opt-env* false)))
		(cond ((and h (eq (car h) 'definer))
		       (!= last-entity-defined* (cadr def-form))
		       (funcall (cadr h) def-form))
		      (t
		       (values
			  `(define ,(flagexp "Undefinable" (cadr def-form))
				 ,@(cddr def-form))
			  false)))))
	    (t
	     (values
		`(define ,(flagexp "Unintelligible" (cdr def-form)))
		false)))))

(declaim (special aux-defs-name*))

(defmacro define (&rest stuff)
   (bind ((aux-defs* '())
	  (eval-aux-defs* false)
	  (auxfn-counter* 0)
	  (aux-defs-name* (extract-aux-defs-name stuff)))
      (opt-define stuff)
      `(progn (bind ((aux-defs* ':not-allowed)
		     (eval-aux-defs* false)
		     (auxfn-counter* 0)
		     (aux-defs-name* ',aux-defs-name*))
		 (opt-define ',stuff))
	      ,@(nisptype::aux-defs-global aux-defs*)
	      ',(or aux-defs-name*
		   (cadr (cadr stuff))))))

;;; Because not all names are global, this is somewhat tricky and
;;; duplicates some of what 'opt-define' does.
;;; Only certain things can define auxiliary functions, so we ignore
;;; the others.
(defun extract-aux-defs-name (definer)
   (match-cond definer
      ?( ((?(:\| domain problem)
	   ?name)
	  ?@_)
	name)
      ?( ((?(:& ?kind (:\| addendum situation))
	   ?name)
	  ?@fields)
        (multi-let (((status dom _ _)
		     (extract-domain-and-fields fields)))
	   (cond ((eq status ':okay)
		  (build-symbol (< dom) - (< name)))
		 (t
		  (out (:to *error-output*)
		     "Warning: Can't retrieve distinguishing aux-def name"
		     " from definition of " (:a kind) 1 name :%)
		  (build-symbol "???-" name)))))
      (t false)))

(defun opt-define (stuff)
   (let ((flag-count* 0))
      (!= last-entity-defined* false)
      (cond ((and (consp (car stuff))
                  (symbolp (caar stuff)))
             (multi-let (((h _) (opt-form-handler (car stuff) global-opt-env* false)))
                (cond ((and h (eq (car h) 'definer))
                       (let ((ex (funcall (cadr h) `(define ,@stuff))))
                          (cond ((> flag-count* 0)
				 (cerror "I'll proceed"
					 "Flagged ~s subexpression(s) in define ~s"
					 flag-count* (car stuff))
                                 ex)
                                (t
				 (!= last-entity-defined* (car stuff))
				 (caar stuff)))))
                      (t
                       (cerror "I'll ignore it"
                               "No way to define: ~s"
                               stuff)
                       nil))))
            (t
             (cerror "I'll ignore it"
                     "Unintelligible define ~s"
                     stuff)
             nil))))

;;; Make sure domain names are recognizable as such
#+pre-chunk
(datafun to-slurp define
   (defun (e slurp-if-substantive)
      (cond (slurp-if-substantive
	     (match-cond e
		?( (define (domain ?n) ?@_)
		  (try-domain-with-name n true)))))
      true))

(defmacro auxiliary-code (&rest body)
   `(progn ,@body))

(defvar auxiliary-code-print-level* 4)

(def-opt-form-handler auxiliary-code top-level (code-form)
   (repeat :for ((exp :in (cdr code-form))
		(res '()))
      (eval exp)
      (!= res (cons (level-condense exp auxiliary-code-print-level*)
		    *-*))
    :result (values `(auxiliary-code ,@(reverse res))
		    false)))

(def-opt-form-handler in-package top-level (form)
   (let ((pkg (find-package (cadr form))))
      (cond (pkg
	     (setf-during-file-transduction *package* pkg)
	     (values `(in-package ,(intern (package-name pkg) keyword-package*))
		     false))
	    (t
	     (signal-problem in-package
		"Undefined package name " (cadr form))))))

(defun level-condense (x l)
   (cond ((atom x) x)
	 ((= l 0) '--)
	 (t
	  (<# (\\ (y) (level-condense y (- l 1)))
	      x))))