;-*- Mode: Common-lisp; Package: opt; Readtable: ytools; -*-
(in-package :opt)
;;; $Id: qexpdt.lisp,v 1.4 2005/11/27 04:19:41 dvm Exp $

(depends-on %module/ ytools)

(depends-on :at-run-time %opt/ types)

;;;(needed-by-macros
;;;   (export '(Exp-index rehash-spec Expcoord topcoord
;;;	     exptree-init exp-index-init
;;;	     exp-fetch exp-ob-index exp-rehash-key
;;;	     exp-index-subconts exp-index-copy
;;;	     contents tree size Exp-index-contents
;;;	     force-rehash discrim exp-index-see topcoord*)
;;;	   (find-package :opt))
;;;)

(depends-on :at-run-time %langutils/ symboid)

; Discrimination trees of objects indexed by s-expressions they contain

;; Discriminations are made on the basis of the S-expressions
;; appearing at a given position in the data to be discriminated.  A
;; position is called an "expression coordinate" (Expcoord).  A
;; position is printed in the form  #>C(A | D)*R.  
;; Every Expcoord is related to others in a
;; straightforward way.  E.g., for Expcoord #>CAAR,
;;   #>CR
;;    #>CAR                its UP is (#>CAAR #>CAR #>CR)
;;      #>CAAR
;;        #>CAAAR          its DOWNLEFT is #>CAAAR
;;        #>CDAAR          its DOWNRIGHT is #>CDAAR
;;      #>CDAR

(def-class Expcoord (:options :key)
         (:handler
	      (:print-function (c srm k)
		     (declare (ignore k))
		     (format srm "#>C")
		     (do ((cl (Expcoord-up c) (cdr cl)))
			 ((null (cdr cl)))
			(cond ((Expcoord-is-cdr (car cl))
			       (format srm "D"))
			      (t
			       (format srm "A"))))
		     (format srm "R")))
   apply
   is-cdr
   up
   dl
   dr
   is-var)
   
(defun new-Expcoord (c x is-var)
   (let ((newc (make-Expcoord
		  :apply (cond (c
				(let ((last-fcn
					 (cond ((eq x 'd)
						(try-cdr is-var))
					       (t
						(try-car is-var)))))
				   (cond ((null (cdr (Expcoord-up c)))
					  last-fcn)
					 (t
					  (let ((c-apply (Expcoord-apply c)))
					     #'(lambda (x)
						 (funcall
						    last-fcn
						    (funcall c-apply x))))))))
			       (t #'identity))
		  :is-cdr (eq x 'd)
		  :up (if c (Expcoord-up c) '())
		  :dl nil
		  :dr nil
		  :is-var is-var)))
      (push newc (Expcoord-up newc))
      newc))
  
(declaim (inline Expcoord-is-car))

(defun Expcoord-is-car (c) (not (Expcoord-is-cdr c)))
  
(defun Expcoord-downright (c)
    (or (Expcoord-dr c)
	(let ((new (new-Expcoord c 'd (Expcoord-is-var c))))
	  (setf (Expcoord-dr c) new)
	  new)))
  
(defun Expcoord-downleft (c)
    (or (Expcoord-dl c)
	(let ((new (new-Expcoord c 'a (Expcoord-is-var c))))
	  (setf (Expcoord-dl c) new)
	  new)))

(declaim (inline apply-coord))

(defun apply-coord (c x)
   (exp-keyify (funcall (Expcoord-apply c) x)
	       (Expcoord-is-var c))  )

(defun try-car (is-var)
   #'(lambda (x)
      (cond ((or (funcall is-var x) (eq x '*dontcare))
	     '*dontcare)
	    ((consp x)
	     (car x))
	    (t '*na))))

(defun try-cdr (is-var)
   #'(lambda (x)
      (cond ((or (funcall is-var x) (eq x '*dontcare))
	     '*dontcare)
	    ((consp x) 
	     (cdr x))
	    (t '*na))))

(defun exp-keyify (x is-var)
   (cond ((funcall is-var x) '*dontcare)
         ((atom x) (symbolify x))
         (t '*struct)))

(defvar topcoord*)   ; must be set by package using this system.

(defun topcoord (is-var)
   (new-Expcoord false '_ is-var))

(def-class Exp-key (:options (:medium :list))
   coord
   val)

(defun make-exptree-cell (key cell)
    (cons key cell))

(defvar exptree-dbg* nil)
(defvar dofix* nil)
(defvar rehash-imbalance-thresh* 2)
(defvar imbalance-thresh* 0.9)

(defun make-exptree-table (coord)
   (cons coord (vector '*rehash-in-progress
		       t '())))

(defun exptree-table-coord (tb)
   (car tb))

(defun exptree-table-status (tb)
   (aref (cdr tb) 0))

(defun set-exptree-table-status (tb v)
   (setf (aref (cdr tb) 0) v))

(defsetf exptree-table-status set-exptree-table-status)

(defun exptree-table-assqable (tb)
   (aref (cdr tb) 1))

(defun set-exptree-table-assqable (tb v)
   (setf (aref (cdr tb) 1) v))

(defsetf exptree-table-assqable set-exptree-table-assqable)

(defun exptree-table-entries (tb)
   (aref (cdr tb) 2))

(defun set-exptree-table-entries (tb v)
   (setf (aref (cdr tb) 2) v))

(defsetf exptree-table-entries set-exptree-table-entries)

(defun exptree-table-operative (table tree)
    (or (eq (exptree-table-status table) '*rehash-done)
	(progn
	   (exptree-cleanup tree table)
	   nil)))

(defun make-exptree-entry (val tree)
  (cons val tree))

(defun exptree-entry-val (e) (car e))
(defun exptree-entry-tree (e) (cdr e))

(defvar blip-sw* nil)
(defvar rehash-thresh* 2)

(def-class Exptree
          (:handler
	      (:print-function (self srm k)
		    (declare (ignore k))
		    (format srm "#<Exptree ~s/" (Exptree-cell self))
		    (cond ((exptree-first-operative-table self)
			   (format srm "~s"
			      (mapcan #'(lambda (tab)
					   (cond ((exptree-table-operative
						     tab self)
						  (list (exptree-table-coord
                                                           tab)))
						 (t '())))
				      (Exptree-tables self))))
			  (t
			   (format srm "~s objects"
				   (length (Exptree-objects self)))))
		    (format srm ">")))
   cell   ; exptree-cell
   superior   ; Exptree or nil
   objects    ; list of objects here when here is terminal
   tables     ; list of subtables when nonterminal
   rehash-prospects)   ; prospects for adding to tables
         ; If not *UNKNOWN, this is a list of Expcoords that must be nontrivial
         ; in fetch pattern if rehashing should be attempted.
			    
(defun new-Exptree (cell sup)
   (make-Exptree cell sup '() '() '*unknown))

(defun Exptree-table (tr prop)
   (assoc prop (Exptree-tables tr) :test #'eq))

(defun exptree-operative-table (tree coord)
    (let ((table (Exptree-table tree coord)))
      (if (and table (exptree-table-operative table tree))
	  table nil)))

(defun exptree-first-operative-table (tree)
    (do ((tl (Exptree-tables tree) (cdr tl)))
	((or (null tl)
	     (exptree-table-operative (car tl) tree))
	 (cond (tl
;;; This check is wrong, because it's perfectly legit to have an
;;; *alternative* table at the higher level, so long as it's not on the
;;; path to this subtree.  
;;;;		(let ((sup (and check-exptree-bogosity*
;;;;				(Exptree-superior tree))))
;;;;		   (cond (sup
;;;;			  (let ((suptab (Exptree-table
;;;;					     sup
;;;;					     (exptree-table-coord (car tl)))))
;;;;			     (cond ((and suptab 
;;;;					 (eq (exptree-table-status suptab) '
;;;;					     *rehash-done))
;;;;				    (dbg-save tree (tab (car tl)) suptab)
;;;;				    (breakpoint exptree-first-operative-table
;;;;				       "Redundant table: " (car tl))))))))
		(car tl))
	       (t nil)))))

(defun exptree-contents (tree copy)
     (labels ((collect (tree copy)
		(let ((table (exptree-first-operative-table tree)))
		   (if (not table)
		       (cond (copy (copy-list (Exptree-objects tree)))
			     (t (Exptree-objects tree)))
		     (mapcan #'(lambda (e) (collect (exptree-entry-tree e) t))
			     (exptree-table-entries table))))))
	(collect tree copy)))

(defun exptree-empty (dt)
      (let ((table (exptree-first-operative-table dt)))
        (if (not table) 
            (null (Exptree-objects dt))
            (every #'(lambda (e)
		        (exptree-empty (exptree-entry-tree e)))
		   (exptree-table-entries table)))))

;;; Returns number of objects indexed.
(defun exptree-size (tree)
     (let ((table (exptree-first-operative-table tree)))
       (if (not table)
	   (list-length (Exptree-objects tree))
	   (reduce #'(lambda (n entry)
		        (+ n (exptree-size (exptree-entry-tree entry))))
		   (exptree-table-entries table)
		   :initial-value 0))))

(defun exptree-count-subtrees (tree)
      (</ (\\ (tot tab)
	     (cond ((exptree-table-operative tab tree)
		    (</ (\\ (subtot entry)
			   (+ subtot (exptree-count-subtrees
				         (exptree-entry-tree entry))))
			tot
			(exptree-table-entries tab)))
		   (t tot)))
	  1
	  (Exptree-tables tree)))

(declaim (inline exptree-object-add
		 exptree-object-del
		 exptree-table-add
		 exptree-table-del))

(defun exptree-object-add (tr x)
	 (push x (Exptree-objects tr)))

(defun exptree-object-del (tr x)
	 (setf (Exptree-objects tr)
	       (delete x (Exptree-objects tr) :test #'eq :count 1)))

(defun exptree-table-add  (tr tb)
   (push tb (Exptree-tables tr)))

(defun exptree-table-del (tr tb)
	 (setf (Exptree-tables tr)
	       (delete tb (Exptree-tables tr) :test #'eq :count 1)))

(defun exptree-cleanup (tree table)
   (cond ((and dofix* (eq (exptree-table-status table) '*rehash-in-progress))
	  (format *error-output*
		   "***Rendering table for ~s inoperative***~%"
 		   (exptree-table-coord table))
	  (exptree-table-del tree table))))

(defun exptree-init () (new-Exptree '() nil))

(declaim (inline expcoord-used))

(defun expcoord-used (c dt)
    (assoc c (Exptree-tables dt) :test #'eq))

(defstruct rehash-spec
   coordfn 
   piecefn
   topcoord)

(defun rehash-spec-is-var (rs)
      (Expcoord-is-var (rehash-spec-topcoord rs)))

; At the top level, need to store the procedure PIECEFN that extracts
; index pattern from objects.
(def-class Exp-index
        (:options :key)
        (:handler
           (:print-function (c srm k)
                  (declare (ignore k))
                  (format srm "#<Exp-index ~s>"
                          (Exp-index-tree c))))

   tree rehash)

(defun Exp-index-empty (ei)
   (exptree-empty (Exp-index-tree ei)))

(defun Exp-index-contents (ei copy)
      (exptree-contents (Exp-index-tree ei) copy))

(defun exp-index-count-subtrees (ei)
     (exptree-count-subtrees (Exp-index-tree ei)))

(defun exp-index-init (piecefn topcoord)
      (make-Exp-index :tree (exptree-init)
		      :rehash (make-rehash-spec :coordfn #'exp-rehash-key
						:piecefn piecefn
						:topcoord topcoord)))

(defun exp-index-copy (e)
   (labels ((exptree-copy (tr sup)
	       (make-Exptree
		  (Exptree-cell tr)
		  sup
		  (list-copy (Exptree-objects tr))
		  (mapcar #'(lambda (tab) (tab-copy tab sup))
			  (Exptree-tables tr))
		  (Exptree-rehash-prospects tr)))
	    (tab-copy (tab sup)
		(let ((newtab
		          (make-exptree-table
			     (exptree-table-coord tab))))
		   (setf (exptree-table-status newtab)
			 (exptree-table-status tab))
		   (setf (exptree-table-assqable newtab)
			 (exptree-table-assqable tab))
		   (setf (exptree-table-entries newtab)
			 (mapcar #'(lambda (ent)
				      (make-exptree-entry
				         (exptree-entry-val ent)
					 (exptree-copy
					     (exptree-entry-tree ent)
					     sup)))
				 (exptree-table-entries tab)))
		   newtab)))
	 (make-Exp-index
	     :tree (exptree-copy (Exp-index-tree e) nil)
	     :rehash (Exp-index-rehash e))))

; Is every element of e1 in e2 (eq-tested)?
(defun exp-index-subconts (e1 e2)
   (let ((piecefn (rehash-spec-piecefn (Exp-index-rehash e1))))
      (let-fun
              ((:def search1 (subtree)
		  (let ((optab (exptree-first-operative-table subtree)))
		     (cond ((not optab)
			    (every #'find2 (exptree-contents subtree nil)))
			   (t
			    (every (\\ (e)
				      (search1 (exptree-entry-tree e)))
				   (exptree-table-entries optab))))))
	       (:def find2 (b1)
		  (member b1 (exp-fetch (funcall piecefn b1)
					e2 t)
			  :test #'eq)))
	  (search1 (Exp-index-tree e1)))))

(defun exp-ob-index (ob ind add-sw)
   (let ((r (Exp-index-rehash ind)))
      (exptree-index (Exp-index-tree ind)
		     ob (funcall (rehash-spec-piecefn r) ob)
		     add-sw)))

(defun exptree-index (tree thing expr add-sw)
    (labels
	((process (tables found)
	   (cond (tables
		  (let ((tab (car tables)))
		     (case (exptree-table-status (car tables))
			(*rehash-done
			 (exp-table-index tab thing expr add-sw tree)
			 (process (cdr tables) t))
			(*rehash-in-progress
			 (exptree-cleanup tree tab)
			 (process (cdr tables) found))
			(t
			 (cond ((and add-sw
				     (not (eql (apply-coord
						  (exptree-table-coord tab)
						  expr)
					      (exptree-table-status tab))))
                                ; un-nix-key if new thing differs
                                ; at this coord from the dominant symbol
				(exptree-table-del tree tab)
                                (setf (Exptree-rehash-prospects tree)
				      '*unknown)))
			 (process (cdr tables) found)))))
		 (found)
		 (add-sw
		  (cond (blip-sw* (format *error-output* "'")))
		  (cond (exptree-dbg*
			 (format *error-output*
			   "***DT debug: adding ~s to exptree ~s~%" 
			   thing
			   (Exptree-cell tree))))
		  (exptree-object-add tree thing))
		 (t
		  (cond (blip-sw* (format *error-output* "'")))
		  (cond (exptree-dbg*
			 (format *error-output*
			   "***DT debug: removing ~s from exptree ~s~%" 
			   thing
			   (Exptree-cell tree))))
		  (exptree-object-del tree thing)))))
      (process (Exptree-tables tree) nil)))

(defun exp-table-index (table thing expr add-sw tree)
  (let ((entry (coord-entry (exptree-table-coord table) expr table tree)))
    (exptree-index (exptree-entry-tree entry)
		   thing expr add-sw)))

(defun coord-entry (c expr table tree)
   (let ((val (apply-coord c expr)))
      (let ((entry (best-assoc-fetch val table)))
	(or entry
	    (let ((entry (make-exptree-entry val
				 (new-Exptree
				       (make-exptree-cell
					     (make-Exp-key
					         (exptree-table-coord table)
						 val)
					     (Exptree-cell tree))
                                       tree))))
	       (push entry (exptree-table-entries table))
               (cond ((and (exptree-table-assqable table)
                           (not (is-Symboid val)))
                      (setf (exptree-table-assqable table) nil)))
	       entry)))))

(defun exp-fetch (pat ind variant)
   (exptree-fetch pat (Exp-index-tree ind) variant (Exp-index-rehash ind)))

(defun exptree-fetch (pat dt variant rehash)
  (multiple-value-bind
      (table val)
      (choose-exptree-table pat dt variant rehash)
    (if table
	(nconc (let ((entry1 (best-assoc-fetch val table)))
		  (cond (entry1
			 (exptree-fetch
			    pat (exptree-entry-tree entry1) variant rehash))
			(t '())))
	       (cond ((or variant (eq val '*dontcare))
		      '())
		     (t
		      (let ((entry2 (assoc '*dontcare
					   (exptree-table-entries table)
					   :test #'eq)))
			 (cond (entry2
				(exptree-fetch pat (exptree-entry-tree entry2)
					       variant rehash))
			       (t '()))))))
	(exptree-contents dt t))))

(defun best-assoc-fetch (val table)
  (if (exptree-table-assqable table)
      (assoc val (exptree-table-entries table) :test #'eq)
      (assoc val (exptree-table-entries table))))

;; All of the complexity in this algorithm is due to the fact that
;; there is a substantial overhead for realizing that there is no point 
;; in further discrimination ("rehashing") at a dtree node.  
;; After an unsuccessful rehash, we look hard to see if there's ever going
;; to be another chance.
;; REHASH-PROSPECTS is a list of all pieces of last unsuccessful fetch pattern
;; that, if they had been present, might have caused a successful rehash.
;; We hope that in the typical case, this can be shown to be (), so
;; that you never have to think about rehashing this node again.

;; In the old regime, the only way to detect that rehashes were
;; impossible was to try all possible ways and reject them all.
;; Unfortunately, these futile searches were the norm.  If fetching
;; (FOO (F ?X)), and already discriminated on #>CAR, #>CAADR, and
;; determined that the #>CADR doesn't help, then you're done.  Hence we
;; provide two mechanisms for generating Expcoords to discriminate on:
;; a general-purpose, slow keylist generator; and a fast traverser
;; that finds the first key the keylist version would generate, or
;; returns nil if the keylist version would generate nothing.  In the
;; fully-discriminated case, this is the norm, and it should happen as
;; fast as possible.  It's not clear this complexity is still worth
;; the trouble, but it's been left in.

(defvar allow-exptree-rehash* true)

;; Returns subindex plus obj at coord for dt within it
(defun choose-exptree-table (pat dt variant rehash)
  (do ((table)
       (tbls (Exptree-tables dt) (cdr tbls))
       (found nil) (val))
      ((or found (null tbls))
       (cond (found (values table val))
	     (t
	      (let ((rhp (Exptree-rehash-prospects dt)))
		(cond ((and allow-exptree-rehash*
			    (not (null rhp))
			    (or (exptree-first-operative-table dt)
				(>= (exptree-size dt) rehash-thresh*))
			    (or (eq rhp '*unknown)
				(some #'(lambda (c)
					  (let ((p (apply-coord c pat)))
					     (and (not (eq p '*na))
						  (or variant
						      (not (eq p '*dontcare)))
						  (not (key-nixed c p dt)))))
				      rhp)))
		       (exptree-rehash pat dt variant rehash))
		      (t
		       (cond (exptree-dbg*
			      (err-out
				      "No prospect of rehash at "
				      (Exptree-cell dt))))
		       (values nil nil)))))))
      (setq table (car tbls))
      (if (exptree-table-operative table dt)
	(multiple-value-setq
	    (found val)
            (pat-has-key-val pat (exptree-table-coord table) variant)))))

(defun pat-has-key-val (pat c variant)
  (let ((disc-val (apply-coord c pat)))
    (cond ((eq disc-val '*dontcare)
	   (if variant (values t '*dontcare) (values nil '())))
          (t (values t disc-val)))))

;; Pick a new key to break the contents of DT down by.  Create, plug in,
;; and return new subindex for it.  If no key worth rehashing on, return NIL.
; Returns subindex plus dnkeyval for dt within it.
(defun exptree-rehash (pat dt variant rehash)
  (multiple-value-bind (coord val rehash-keys)
		       (funcall (rehash-spec-coordfn rehash)
                               pat dt (rehash-spec-topcoord rehash) variant)
    ; First call fast algorithm to see if there's a candidate key.
    (cond ((not coord)
	   (cond (exptree-dbg*
		  (format *error-output* "Not worth rehashing ~%")))
           (setf (Exptree-rehash-prospects dt)
		 (missing-pieces pat dt (rehash-spec-topcoord rehash)))
	   (values nil nil))
	  (t
	   (cond (exptree-dbg*
		  (err-out "Rehashing exptree (Exptree-cell dt)"
			   :% "  Attempting rehash on coord " coord
			   :% " Objects: " (exptree-contents dt nil))))
	   (let ((dofix* nil))
	     (let* ((things (exptree-contents dt nil))
		    (table (new-exptree-table dt coord)))
	       (cond ((and table
			   (rehash-exptree-things table things dt
						  (rehash-spec-piecefn rehash)
						  val))
		      (values table val))
		     (t
		      ; If the first candidate doesn't work, fall back on
		      ; the complex method.
		      (do ((key)
			   (kl (cond (rehash-keys
				      (funcall rehash-keys
					       pat dt
					       (rehash-spec-topcoord rehash)
					       variant))
				     (t '()))
			       (cdr kl))
			   (table) (pos))
			  ((null kl)
			   (progn
			      (cond (exptree-dbg*
				     (format *error-output*
					     "No more keys to rehash on ~
					       -- rehashing failed")))
			      (setf (Exptree-rehash-prospects dt)
				    (missing-pieces
				       pat dt
				       (rehash-spec-topcoord rehash)))
			      (values nil nil)))
			  (setf key (car kl))
			  (setf pos (Exp-key-coord key))
                          (setf table
			      (new-exptree-table dt pos))
			  (cond ((and table exptree-dbg*)
				 (format *error-output*
                                   "Attempting rehash on coord ~s~%"
				   pos)))
			  (cond ((and table
				      (rehash-exptree-things
				         table things dt
					 (rehash-spec-piecefn rehash) 
					 (Exp-key-val key)))
				 (return (values table (Exp-key-val key))))
			  ))))))))))

(defun missing-pieces (pat et topcoord)
   (let ((is-var (Expcoord-is-var topcoord)))
      (labels ((collect-pieces (pat c)
		 (cond ((funcall is-var pat)
			(cond ((key-useless-here c et) '())
			      (t (list c))))
		       ((atom pat) '())
		       (t
			(do ((pl pat (cdr pl))
			     (r '()))
			    ((null pl)
			     r)
			  (setf r (nconc (collect-pieces
					     (car pl)
					     (Expcoord-downleft c))
					r))
			  (setf c (Expcoord-downright c)))))))
	 (collect-pieces pat topcoord))))

; Try to rehash IND so that a fetch for PAT won't find AVOID
; Return t if successful.  PAT and AVOID differ in value at 
; coord DIF
(defun force-rehash (pat avoid dif variant ind)
   (exptree-force pat avoid dif 
                  (Exp-index-tree ind) variant (Exp-index-rehash ind)))

(defun exptree-force (pat avoid dif dt variant rehash)
  (multiple-value-bind
      (table val)
      (choose-exptree-table pat dt variant rehash)
       ; general CHOOSEr is not needed, but probably won't hurt.
    (if table
	(or (let ((entry1 (best-assoc-fetch val table)))
	       (cond (entry1
		      (exptree-force
			    pat avoid dif
			    (exptree-entry-tree entry1)
			    variant rehash))
		     (t nil)))
	    (cond (variant nil)
		  (t
		   (let ((entry2 (assoc '*dontcare
					(exptree-table-entries table)
					:test #'eq)))
		      (cond (entry2
			     (exptree-force pat avoid dif
					    (exptree-entry-tree entry2)
					    variant rehash))
			    (t nil))))))
	(cond ((member avoid (exptree-contents dt nil) :test #'eq)
               (setf table (new-exptree-table dt dif))
	       (cond (table
		      (let ((piecefn (rehash-spec-piecefn rehash)))
			 (dolist (thing (exptree-contents dt nil))
			    (exp-table-index table thing (funcall piecefn thing)
					     t dt)))
		      (setf (Exptree-objects dt) '())
		      (setf (exptree-table-status table) '*rehash-done)
		      (setf (Exptree-rehash-prospects dt) '*unknown)
		      (let ((sub (best-assoc-fetch (apply-coord dif pat)
						   table)))
			 (or (not sub)
			     (not (member avoid
					  (exptree-contents (exptree-entry-tree sub)
							    nil)
					  :test #'eq)))))
		     (t
		      (signal-problem exptree-force
			 "Can't add table for " dif " to " dt))))
              (t nil)))))

(defun new-exptree-table (tree coord)
  (let ((table (expcoord-used coord tree)))
    (cond ((not table)
	   (cond ((assq (exptree-table-coord table)
			(Exptree-cell tree))
		  false)
		 (t
		  (let ((table (make-exptree-table coord)))
		    (exptree-table-add tree table)
		    table))))
	  ((eq (exptree-table-status table) '*rehash-in-progress)
	   (cerror "I'll ignore it"
                   "Table's status already indicates *rehash-in-progress")
	   (setf (exptree-table-entries table) '())
	   table)
	  (t (setf (exptree-table-status table) '*rehash-in-progress)
	     (setf (exptree-table-entries table) '())
	     table))))

(defun rehash-exptree-things (table things tree piecefn val)
  (dolist (thing things)
     (exp-table-index table thing (funcall piecefn thing) t tree))
;  (out (to (errout))
;       "testing rehash, val = " val
;       t " entries = "
;       :% (exptree-table-entries table) :%
;       :% "  Imbalance threshold = "
;       (* imbalance-thresh* (list-length things)) :%)
  (do ((thresh (* imbalance-thresh* (list-length things)))
       (entries (exptree-table-entries table) (cdr entries))
       (entry))
      ((null entries)
       (progn
	  (cond (exptree-dbg*
		 (format *error-output* "Rehashing succeeded ~%")))
	  (setf (Exptree-objects tree) '())
	  (setf (exptree-table-status table) '*rehash-done)
	  (setf (Exptree-rehash-prospects tree) '*unknown)
	  table))
    (setf entry (car entries))
;    (out (to (errout))
;	 "Next entry: " entry
;	 :% " entry-val = " (exptree-entry-val entry)
;	 :% " entry-size = " (exptree-size (exptree-entry-tree entry))
;	 :%)
    (cond ((and (eql val (exptree-entry-val entry))
		(>= (exptree-size (exptree-entry-tree entry))
		    thresh))
	   (cond (exptree-dbg*
		  (format *error-output*
		      "Rehashing failed to discriminate -- undoing it~%")))
	     (setf (exptree-table-status  table) (exptree-entry-val entry))
	     (setf (exptree-table-entries table) '())
	     (return nil)))))

(defun exp-rehash-key (pat dt topcoord variant)
  (multiple-value-bind (c b)
		       (first-rehash-key pat topcoord dt variant)
     (values c b #'exp-rehash-keys)))

(defun listrehashkey (pat c dt skip-car variant)
  (let ((pat  (if skip-car (cdr pat) pat))
	(c    (if skip-car (Expcoord-downright c) c)))
     (cond ((or (atom pat) (funcall (Expcoord-is-var c) pat))
            (values nil nil))
	   (t
	    (do ((x)
		 (pl pat (cdr pl))
		 (coord) (val))
		((null pl)
		 (values nil nil))
	      (setf x (car pl))
	      (multiple-value-setq (coord val)
				   (first-rehash-key x (Expcoord-downleft c)
						     dt variant))
	      (cond (coord
		     (return (values coord val))))
	      (setf c (Expcoord-downright c)))))))

(defun first-rehash-key (pat c dt variant)
   (cond ((funcall (Expcoord-is-var c) pat) 
          (cond (variant
                 (cond ((key-nixed c '*dontcare dt)
                        (values nil nil))
                       (t
                        (values c '*dontcare))))
                (t (values nil nil))))
         ((atom pat)
          (setf pat (symbolify pat))
          (cond ((key-nixed c pat dt)
                 (values nil nil))
                (t (values c pat))))
         ((key-nixed c '*struct dt)
          (listrehashkey pat c dt nil variant))
         (t
          (values c '*struct))))

(defun exp-rehash-keys (pat et topcoord variant)
  (keygen pat topcoord et variant))

;; Generates a list of keys for a pat appearing at Expcoord.
(defun keygen (pat c et variant)
  (cond ((funcall (Expcoord-is-var c) pat)  
	 (cond ((and variant (not (key-nixed c '*dontcare et)))
		(list (make-Exp-key c '*dontcare)))
	       (t '())))
	((atom pat)
         (cond ((key-nixed c pat et) '())
               (t (list (make-Exp-key c (symbolify pat))))))
	((key-nixed c '*struct et)
         (listkeygen pat c et variant))
        (t
	 (cons (make-Exp-key c '*struct)
	       (listkeygen pat c et variant)))))

(defun listkeygen (pat c et variant)
  (if (null pat)
      '()
      (nconc (keygen (car pat) (Expcoord-downleft  c) et variant)
	     (listkeygen (cdr pat) (Expcoord-downright c) et variant))))


;; Returns coord where they differ, or false if they don't differ (as
;; far as discrim can figure out).
(defun discrim (pat1 pat2 topcoord as-variants)
   (let ((is-var (Expcoord-is-var topcoord)))
      (labels ((find-diff (pat1 pat2 c)
;;;;		 (trace-around find-diff
;;;;		    (:> "(find-diff: " :% 5 pat1 :% " vs. " pat2")")
		 (cond ((or (funcall is-var pat1)
			    (funcall is-var pat2))
			(cond (as-variants
			       (cond ((and (funcall is-var pat1)
					   (funcall is-var pat2))
				      false)
				     (t c)))
			      (t false)))
		       ((or (atom pat1) (atom pat2))
;;;;			(strip-sym-with-type pat1)
;;;;			(strip-sym-with-type pat2)
			(cond ((eq pat1 pat2) nil)
			      (t c)))
		       (t
			(do ((pl1 pat1 (cdr pl1))
			     (pl2 pat2 (cdr pl2))
			     (r))
			    ((or (null pl1) (null pl2))
			     nil)
			   (setf r (find-diff (car pl1) (car pl2)
					      (Expcoord-downleft c)))
			   (cond (r
				  (return r)))
			   (setf c (Expcoord-downright c)))))
;;;;		    (:< (val &rest _) "find-diff: " val))
		 ))
	 (find-diff pat1 pat2 topcoord))))

(defun key-nixed (k val dt)
   (do ((c '*here))
       ((let ((tab (expcoord-used-before k dt c)))
	  (and tab
	       (or (eql (exptree-table-status tab) val)
		   (eq (exptree-table-status tab) '*rehash-done))))
	t)
      (setf c (cond ((eq c '*here) (Exptree-cell dt))
		    (t (cdr c))))
      (setf dt (Exptree-superior dt))
      (cond ((not dt)
	     (return nil)))))

; Test whether K will never be of any use because it's already been used
; to discriminate.  Only count successful rehashes, because unsuccessfuls
; can be retried after future additions.
(defun key-useless-here (k dt)
   (do ((c '*here))
       ((let ((tab (expcoord-used-before k dt c)))
	  (and tab (eq (exptree-table-status tab) '*rehash-done)))
	t)
      (setf c (cond ((eq c '*here) (Exptree-cell dt))
                  (t (cdr c))))
      (setf dt (Exptree-superior dt))
      (cond ((not dt)
	     (return nil)))))

; Get table for Expcoord POS at ET, *if* it occurs before subtree
; corresponding to cell C.  
(defun expcoord-used-before (pos et c)
   (let ((check (and c
                     (not (eq c '*here))
                     (Exp-key-coord (car c)))))
      (dolist (tab (Exptree-tables et)
		   nil)
         (cond ((eq (exptree-table-coord tab) check)
		(return (cond ((eq pos check)
			       tab)
			      (t nil))))
	       ((eq (exptree-table-coord tab) pos)
		(return tab))))))

(declaim (special keyword-package*))

;; Make A into an object that can be tested using EQ or EQL
(defun symbolify (a)
  (cond ((null a) a)
        ((or (is-Symboid a)
	     (numberp a)
	     (characterp a)
	     (is-Canonized-symbol a))
	 a)
;;;;	((is-Sym-with-type a) (Sym-with-type-actual a))
        ((stringp   a) (intern a keyword-package*))
	(t a)))

(defun exp-index-see (ei depth)
   (exptree-see (Exp-index-tree ei) depth 0 nil
		(rehash-spec-piecefn (Exp-index-rehash ei))))

(defun exptree-see (tree depth indent mid piecefn)
     (cond ((not mid)
	    (format t "~&")
	    (print-spaces t indent)
	    (format t "Cell ~s" (Exptree-cell tree))))
     (cond ((not (eq (Exptree-rehash-prospects tree) '*unknown))
	    (format t " [Rehash prospects: ~s]"
		    (Exptree-rehash-prospects tree))))
     (if (not (exptree-first-operative-table tree))
	 (let ((i (+ indent 3)) (l (exptree-contents tree nil)))
	    (dolist (a l)
	       (format t "~&")
	       (print-spaces t i)
	       (let ((*print-pretty* t))
		  (format t "~s"
			  (funcall piecefn a))))))
     (let ((k (list-length (Exptree-tables tree))))
        (format t "~%")
	(print-spaces t indent)
	(format t " -- ~s tables~%" k)
	(dolist (table (Exptree-tables tree))
	  (exptree-table-see table depth (+ indent 3) piecefn))))

(defun exptree-table-see (table depth indent piecefn)
  (let ((coord (exptree-table-coord table)))
    (case (exptree-table-status table)
      (*rehash-in-progress
       (format t "~&")
       (print-spaces t indent)
       (format t "~s : Rehash in progress ~%"
	       coord))
      (*rehash-done
       (dolist (entry (exptree-table-entries table))
	  (format t "~&")
	  (print-spaces t indent)
	  (cond ((eq (exptree-entry-val entry) '*na)
		 (format t "~s discrimination not applicable"
			 coord))
		(t
		 (format t "~s: ~s" coord (exptree-entry-val entry))))
	  (cond ((> depth 0)
		 (exptree-see (exptree-entry-tree entry)
			      (- depth 1)
			      (+ indent 3)
			      t piecefn))
		(t (print-spaces t 3 )
		   (format t "...")))
	  (format t "~%")
	  (print-spaces t indent)
	  (format t "------------")))
      (t
       (format t "~&")
       (print-spaces t indent)
       (format t "~s: Rehash failed due to frequency of: ~s~%"
	       coord (exptree-table-status table))))))


; For debugging
(defun strip-nixes (et)
   (setf (Exptree-rehash-prospects et) '*unknown)
   (setf (Exptree-tables et) 
       (remove-if-not
	   #'(lambda (tab) (eq (exptree-table-status tab) '*rehash-done))
           (Exptree-tables et)))
   (dolist (tab (Exptree-tables et))
      (dolist (e (exptree-table-entries tab))
         (strip-nixes (exptree-entry-tree e)))))

(defun print-spaces (srm num)
  (dotimes (i num)
     #-(or :cmu :allegro) (declare (ignore i))
     (format srm " ")))

