;-*- Mode: Common-lisp; Package: opt; Readtable: ytools; -*-
(in-package :opt)
;;; $Id: qunify.lisp,v 1.35 2005/11/28 14:58:32 dvm Exp $

(depends-on %module/ ytools)
(depends-on :at-run-time %opt/ types)

(depends-on :at-run-time %langutils/ symboid)

(depends-on (:at :slurp-time) %ytools/ bq)

(eval-when (:compile-toplevel :load-toplevel :execute :slurp-toplevel)
   (export '(unify variants qexps-equal)))

(end-header :continue-slurping)

;;; This unifier renames variables in constant time by defining a
;;; variable as a pair <symbol, varid>, where varid is an arbitrary integer.
;;; To rename all the variables in an expression, just generate a new varid.
;;; We no longer require, when a variable with id I is bound to an 
;;; expression with id J, that I>=J.
;;; We no longer unify step-value's by using an "optimistic unification" 
;;; mechanism. 

(eval-when (:compile-toplevel :load-toplevel :execute)
   (defconstant +max-varid+ (- most-positive-fixnum 10))
   (deftype Varid () `(integer 0 ,+max-varid+)))
;;; nil is no longer a legal Varid; its place is now taken by 0.


(yt::subr-synonym varid= =)
(define-compiler-macro varid= (id1 id2)
    `(= (the fixnum ,id1) (the fixnum ,id2)))

;;; The name is purely for debugging; two distinct Boundvars can have the same
;;; name.  (This should move to internalize.)
;;; The point is that internalize must make sure that the Boundvars
;;; for a Binder Qexp occur in no other Binder Qexp.
(def-class Boundvar name
   (:handler
       (print (lv srm)
	  (out (:to srm)
	    (:a "\\") (Boundvar-name lv) (:a "\\") ))))

;;; No need for this type; any object type that is eq-testable can serve
;;; as a varname.
;;;(deftype Uvar () '(or symbol Symboid Boundvar ))

(deftype Sexp ()
   '(or symbol string number
       (and cons (satisfies
		    (lambda (l)
		       (is-list-of l (\\ (e) (typep e 'Sexp))))))))

(needed-by-macros

(def-class Qexp-handler (:options :key)
   (name nil :type symbol)
   (global-var nil :type symbol)
   (upriority 0 :type fixnum) ;; Integer priority for --
   (unify-handler nil (function (Qexp Qexp Varid Varid Bdgenv)
				(values boolean Bdgenv)))
     ;; -- unification handler
   variant-handler ;; variant-check handler
   occur-checker ;; handler to do the occur-check
   (index-pat-maker ;; create pattern to use in indexing
      (\\ (_) nil)
      (function (Qexp) t))  ;; The result is an S-expression
   sexp-maker    ;; transform to S-expression
   (main-op-extractor  ;; Extract symbol describing head, plus
                       ;; args as a list of Qexps, plus adjusted env
		       ;; so that free vars in args list
		       ;; are all w.r.t. targ-id.  For short
		       ;; qexps, second val is a flag specifying
		       ;; what head is.
       (\\ (qe qe-id targ-id env)
	  (ignore qe qe-id targ-id)
	  (values t nil env))
       (function (Qexp Varid Varid Bdgenv) (values symbol t Bdgenv)))

;; Return head + args
   (long nil boolean)          ;; whether Qexps with this handler are long
   )
)
;;; 'index-pat-maker' transforms a Qexp into an S-expression to use in
;;; indexing.  Key features: Qvars become *dontcare's; list-qexps become
;;; (cons e1 (cons e2 ....(cons eN tl))).

(defvar qexp-class-tab* (make-hash-table :test #'eq))

(eval-when (:compile-toplevel :execute :slurp-toplevel)
   (defun qexp-class-var (classname)
      (build-symbol (:package :opt) (< classname) -qexp-class*)))

(defmacro def-qexp-class (name &key long
				    ((:upriority upri^))
				    ((:unify-handler uhandler^))
				    ((:variant-handler vhandler^))
				    ((:occur-checker occur-checker^))
				    ((:index-pat-maker index-pat-maker^))
				    ((:sexp-maker sexp-maker^))
				    ((:main-op-extractor main-op-extractor^)))
   (let-fun ()
      (repeat :for ((hv :in '(:unify-handler :variant-handler :occur-check
			      :index-pat-maker :sexp-maker :main-op-extractor))
		    (v :in (list uhandler^ vhandler^ occur-checker^
				 index-pat-maker^ sexp-maker^ main-op-extractor^)))
	 (cond ((not v)
		(signal-problem def-qexp-class
		   "Must supply " hv " for qexp class " name))))
      (!= uhandler^ (insert-decls *-* '(Qexp Qexp Varid Varid Bdgenv)))
      (!= vhandler^
	  (insert-decls *-* '(Qexp Qexp Varid Varid Bdgenv Bdgenv)))
      (!= index-pat-maker^
	  (insert-decls *-* '(Qexp)))
      (!= name (intern *-* keyword-package*))
      (let ((class-var-name (qexp-class-var name)))
	 `(defvar ,class-var-name 
		     (make-Qexp-handler
			:name ',name
			:global-var ',class-var-name
			:long ,long
			:upriority ,upri^
			:unify-handler ,uhandler^ ,
			:variant-handler ,vhandler^
			:occur-checker ,occur-checker^
			:index-pat-maker ,index-pat-maker^
			:sexp-maker ,sexp-maker^
			:main-op-extractor ,main-op-extractor^)))))

(needed-by-macros

;;; Add declarations to a lambda expression --
(defun insert-decls (defn param-types)
      (match-cond defn
	 (:? ?(:\| (?(:\| \\ lambda) ?params
		    ?@body)
		  #'(lambda ?params
	              ?@body))
	    (cond ((and (exists (p :in params) (not (eq p '_)))
			(not (matchq ((declare ?@_) ?@_)
				     body)))
		   `(\\ ,params
		       (declare ,@(<# (\\ (ty p) `(type ,ty ,p))
				      param-types
				      params))
		       ,@body))
		  (t
		   `(\\ ,params ,@body))))
	 (t
	  ;; It's a form that evaluates to a function, so assume
	  ;; the declarations were inserted properly
	  defn)))
)

;;;;(defun find-qexp-class (name)
;;;;   (or (table-entry qexp-class-tab*)
;;;;       (signal-problem find-qexp-class
;;;;	  "Undefined Qexp class: " name)))

(def-class Qexp (:options :key)
   (handler nil :type Qexp-handler)
   type
   ;; -- Type or false if the entity is a typeless S-expression
   ;; such as 'start' in (at start ...) [a Durham-timed-typed-exp]
   head  ;; function symbol or something similar
  (:handler
      (print (qe srm)
	 (out (:to srm)
	      "#{Q/"
		 (Qexp-handler-name (Qexp-handler qe))
		 "/"
		 (Qexp-type qe)
		 "/"
		 (Qexp-sexp qe) "}"))))

(defun Qexp-classname (q) (Qexp-handler-name (Qexp-handler q)))

(defun Qexp-is-long (qexp)
   (Qexp-handler-long (Qexp-handler qexp)))

(defun Qexp-sexp (q) (funcall (Qexp-handler-sexp-maker (Qexp-handler q))
			      q))

(def-class Long-Qexp (:options :key (:include Qexp))
   (length 0 :type (integer 0 1000))
   args  ;; typically a list of Qexps, but can be something else
   freevars ;; free qvars & boundvars, a list of Symbols and such
   (index-pat false Sexp) ;; S-expression "equivalent" for indexing purposes
)
;;; The 'freevars' should have the property that a "top-level" Qexp has no
;;; bound variables free.  I.e., it's the responsibility of any module that
;;; peels off the bound variables of a 'binder' Qexp to bind them to new Qvars 
;;; (or something else, but not to other boundvars; oops, there's at least
;;; one exception, namely, during unification; damn).

(declaim (type (function (Long-Qexp Long-Qexp) (integer -1000 1000))
	       Long-Qexp-length-dif))

(defun long-qexps-length-dif (lq1 lq2)
   (- (Long-Qexp-length lq1) (Long-Qexp-length lq2)))

(defun long-qexps-same-length (lq1 lq2)
   (= (Long-Qexp-length lq1) (Long-Qexp-length lq2)))

(defmacro build-Qexp (classname head^ 
		      &key ((:args args^) nil args-supplied)
			   ((:type type^) nil)
		      &whole bqe)
			   ;;;;((:sexp sexp^) nil sexp-supplied)
   (let ((headvar (gen-var 'qexp-head))
	 (class-def-var (qexp-class-var classname)))
      (let ((classdef (symbol-value class-def-var)))
	 (let ((long (Qexp-handler-long classdef)))
	    (cond ((not type^)
		   (signal-problem build-Qexp
		      "build-Qexp called without :type argument: "
		      bqe))
		  ((boole-eq long args-supplied)
		   (let ((argvar (and long (gen-var 'qexp-args))))
		      `(let ((,headvar ,head^)
			     ,@(include-if long `(,argvar ,args^)))
			  (,(cond (long
				   'make-Long-Qexp)
				  (t
				   'make-Qexp))
			   :head ,headvar
			   :handler ,class-def-var
			   :type ,type^
			   ,@(cond (long
				    `(:args ,argvar
				      :length (length ,argvar)
				      :freevars (qexps-freevars ,argvar)))
				   (t '()))
;;;;			   :sexp ,(cond (sexp-supplied sexp^)
;;;;					(t !`1!`2(,2,1#headvar
;;;;						  ,1@(cond (long
;;;;							    !`1(<# Qexp-sexp
;;;;								   ,1#argvar))
;;;;							   (t '())))))
			   ))))
		  (args-supplied
		   (signal-problem build-Qexp
		      "Can't build Qexp of class " classname
		      "  with args: " args^))
		  (t
		   (signal-problem build-Qexp
		      "Can't build Qexp of class " classname
		      " without args")))))))

(defun qexps-freevars (qexps)
   (nodupq
      (<! (\\ (qe)
	     (cond ((is-Long-Qexp qe)
		    (list-copy (Long-Qexp-freevars qe)))
		   ((memq (Qexp-classname qe)
			  '(:qvar :boundvar))
		    (list (Qexp-head qe)))
		   (t !())))
	  qexps)))

(defvar vid* 0)

(declaim (type Varid vid*))

(declaim (ftype (function () Varid) new-Varid))

(defun new-Varid ()
   (!= vid* (+ vid* 1))
   (cond ((> vid* +max-varid+)
	  (signal-problem new-Varid
	     "Too many varids"
	     (:proceed "I'll reuse them, which will probably be okay"))
	  (!= vid* 10000)))
   (the Varid vid*))

;;; Id to use when you're sure the pattern has no variables
(defconstant +ground-id+ 0)

(declaim (inline varid-is-ground))
(defun varid-is-ground (id) (= id +ground-id+))

;;; Other useful varid's for one-off circumstances.
(defvar dummy-id* (new-Varid))
(defvar dummy-id2* (new-Varid))

(def-class Varbdg 
   (varname nil) ;; Type is anything eq-testable
   (vid 0 :type Varid)
   (skel nil :type Qexp) ; a Qexp.  "skel" means "skeleton"; it needs 
   (skelid 0 :type Varid)  ; the skelid and an env to get the whole value.
 (:handler
     (print-object (b srm)
	(out (:to srm)
	  "#<Varbdg " (Varbdg-varname b) "^" (Varbdg-vid b)
	  " / " (Varbdg-skel b) (:q ((not (varid= (Varbdg-skelid b)
                                                  +ground-id+))
				     "^" (Varbdg-skelid b))) ">"))))

(declaim (inline vars=))
(declaim (ftype (function (symbol symbol Varid Varid)
			  boolean)
		vars=))

(defun vars= (v1 v2 i1 i2)
   (declare (type Varid i1 i2))
   (and (eq v1 v2) (varid= i1 i2)))

(defun is-list-of-varbdgs (l) (is-list-of l #'is-Varbdg))

(deftype Bdgenv () '(and list (satisfies is-list-of-varbdgs)))

(declaim (inline empty-env env-is-empty))
(defun empty-env () '())
(defun env-is-empty (e) (null e))

(declaim (inline Bdgenv-bdgs))
(defun Bdgenv-bdgs (e) e)

(declaim (inline make-Bdgenv))
(defun make-Bdgenv (e) e)

; Not really a bdgenv; used as flag to tell caller how match failed.
(defvar discrim-bdgs* (list (make-Varbdg 'discrim 0 'discrim false)))

;;; Bdgenvs are always sorted in decreasing 'vid' order.
(defun cons-Bdgenv (var val vid valid e)
   (declare (type Varid vid valid))
   (let ((new-bdg (make-Varbdg var vid val valid)))
      (let-fun ((insert-in-order (e-tail)
		   (cond ((null e-tail)
			  (list new-bdg))
			 (t
			  (let ((b (car e-tail)))
			     (let ((b-vid (Varbdg-vid b)))
				(cond ((< b-vid vid)
				       (cons new-bdg e-tail))
				      ((and (varid= b-vid vid)
					    (eq (Varbdg-varname b)
						var))
				       (cons new-bdg (cdr e-tail)))
				      (t
				       (cons b (insert-in-order (cdr e-tail)))))))))))
	 (insert-in-order e))))

;;; Bind a bunch of variables, all with same id, to a bunch of values, all
;;; with same id.  Assume there are no previous bindings of the vars.
(defun group-bind (vars vals vid valid env)
   (multi-let (((bdgs-before bdgs-after)
		(repeat :for ((b :in (Bdgenv-bdgs env) :tail et)
			      :collector bdgs-before)
		 :until (=< (Varbdg-vid b)
			    vid)
		 :collect b
		 :result (values bdgs-before et))))
      (make-Bdgenv
	 (nconc bdgs-before
		(<# (\\ (var val) (make-Varbdg var vid val valid))
		    vars vals)
		bdgs-after))))

(defun uvar-lookup (name id env)
   (declare (type Varid id))
   (repeat :for ((b :in env))
    :result false
    :until (vars= (Varbdg-varname b)
		  name
		  (Varbdg-vid b)
		  id)
    :result b))

; Find a binding of a variable with eid in env whose value
; is variable with given name and id.  
(defun uvar-lookup-backwards (name id eid env)
   (declare (type Varid id eid))
   (repeat :for ((b :in env))
    :result false
    :until (and (varid= (Varbdg-vid b)
		    eid)
		(let ((vsk (Varbdg-skel b))
		      (vid (Varbdg-skelid b)))
		   (and (varid= vid id)
			(is-Qvaroid vsk)
			(eq (Qvar-sym vsk)
			    name))))
    :result b))

;;; Used only to wrap qvar values that haven't been occur-checked
;;; yet.
(def-class Not-occur-checked
   exp)

(defvar unify-count* 0)
(defvar unify-success-count* 0)

; t1 and t2 are Qexps, e1 and e2 are varids, e is bdgenv
(defun unify (t1 t2 e1 e2 e)
   (declare (type Qexp t1 t2)
	    (type Varid e1 e2)
	    (type Bdgenv e))
   (!= unify-count* (+ *-* 1))
   (multi-let (((ok e)
		(qexp-unify t1 t2 e1 e2 e)))
      (cond (ok
	     ;; Do postponed occur-checks
	     (repeat :for ((b :in (Bdgenv-bdgs e))
			   :collector good-bdgs)
	      :within
		(cond ((is-Not-occur-checked (Varbdg-skel b))
		       (let ((skel (Not-occur-checked-exp (Varbdg-skel b))))
			  (let ((outcome (occur-check
					    (Varbdg-varname b)
					    skel
					    (Varbdg-vid b)
					    (Varbdg-skelid b)
					    e)))
			  (:continue
			   :until (memq outcome '(*circular *has-boundvars))
			   :result (values false e)
			   :when (not (eq outcome '*self))
			   :collect (make-Varbdg
				       (Varbdg-varname b)
				       (Varbdg-vid b)
				       skel
				       (cond ((eq outcome '*has-vars)
					      (Varbdg-skelid b))
					     (t +ground-id+)))))))
		      (t
		       (:continue
			:collect b)))
	      :result (progn (!= unify-success-count* (+ *-* 1))
			     (values true (make-Bdgenv good-bdgs)))))
	    (t
	     (values false e)))))

(defun qexp-unify (t1 t2 e1 e2 e)
   (declare (type Qexp t1 t2)
	    (type Varid e1 e2)
	    (type Bdgenv e))
   (let ((h1 (Qexp-handler t1))
	 (h2 (Qexp-handler t2)))
      (let ((p1 (Qexp-handler-upriority h1))
	    (p2 (Qexp-handler-upriority h2))
	    (u1 (Qexp-handler-unify-handler h1))
	    (u2 (Qexp-handler-unify-handler h2)))
	 (cond ((> p2 p1)
		(funcall u2 t2 t1 e2 e1 e))
	       (t
		(funcall u1 t1 t2 e1 e2 e))))))

;;; Does _not_ check that l1 and l2 are the same length; stops 
;;; when the shorter runs out.
(defun list-unify (l1 l2 id1 id2 e)
   (repeat :for ((x1 :in l1)
		 (x2 :in l2))
    :result (values true e)
    :within
       (multi-let (((ok e1)
		    (qexp-unify x1 x2 id1 id2 e)))
	  (:continue
	   :while ok
	   :result (values false e1)
	      (!= e e1)))))

;;; Most classes are defined in qexp-classes.lisp

(eval-when (:compile-toplevel :load-toplevel :slurp-toplevel :execute)

(defvar +qvar+ (printable-as-string "<qexp is unbound qvar>"))
(defvar +boundvar+ (printable-as-string "<qexp is unbound boundvar>"))
	     
;;; 'qv' is either a qvar Qexp or a boundvar Qexp
(defun qvar-drill-for-main-op (qv cur-id targ-id env)
   (let ((b (uvar-lookup (Qexp-head qv) cur-id env)))
      (cond (b
	     (qexp-main-op-with-id
		(Varbdg-skel b) (Varbdg-vid b)
		targ-id env))
	    ((varid= cur-id targ-id)
	     (values (Qexp-head qv)
		     (cond ((eq (Qexp-classname qv) ':qvar)
			    +qvar+)
			   (t
			    +boundvar+))
		     env))
	    (t
	     (multi-let (((newvar newenv)
			  (new-dummy-var qv cur-id targ-id env)))
		(values newvar !() newenv))))))

;;; head is sym of qvar
(def-qexp-class qvar
   :upriority 100
   :long false
   :unify-handler
      (\\ (qv pat idv idp env)
	 (!= qv (Qexp-head qv))
	 (let ((bdg (uvar-lookup qv idv env)))
	    (cond (bdg
		   (let ((skel (Varbdg-skel bdg)))
		      (cond ((is-Not-occur-checked skel)
			     (!= skel (Not-occur-checked-exp skel))
			     ;; Can't postpone any longer
			     (let ((outcome (occur-check
					       qv skel
					       idv (Varbdg-skelid bdg)
					       env)))
			       (ecase outcome
				  (*self
				   ;; Old binding is irrelevant
				   (values true
					   (cons-Bdgenv
					      qv (maybe-not-occur-checked pat)
					      idv idp env)))
				  ((*circular *has-boundvars)
				   (values false env))
				  ((*const *has-vars)
				   (let ((skelid 
					    (cond ((eq outcome '*const)
						   +ground-id+)
						  (t (Varbdg-skelid bdg)))))
				      (qexp-unify skel pat
						  skelid idp
						  (cons-Bdgenv
						     qv skel
						     idv +ground-id+ env)))))))
			    (t
			     (qexp-unify (Varbdg-skel bdg)   pat
					 (Varbdg-skelid bdg) idp
					 env)))))
		  (t
		   (values true
			   (cons-Bdgenv qv (maybe-not-occur-checked pat)
					idv idp env))))))

   :variant-handler
      (\\ (var pat id-var id-pat pairings e)
	 (!= var (Qexp-head var))
	 (let ((bdg (uvar-lookup var id-var e)))
	    (cond (bdg 
		   (qexp-variants (Varbdg-skel bdg)
				  pat
				  (Varbdg-skelid bdg)
				  id-pat
				  pairings e))
		  (t
		   (repeat
		    :while (is-Qvar pat)
		    :result (values false pairings discrim-bdgs*)
		    :within
		       (let ((pb (uvar-lookup (Qvar-sym pat)
					      id-pat e)))
			  (:continue
			   :while pb
			   :result (pair-and-continue)
			      (!= pat (Varbdg-skel pb))
			      (!= id-pat (Varbdg-skelid pb))))
		    :where
		       (:def pair-and-continue ()
			  (!= pat (Qvar-sym pat))
			  (repeat :for ((pair :in pairings))
			   :result (values
				      true
				      (cons-Bdgenv
					 var id-var pat id-pat pairings)
				      false)
			   :within
			      (let ((v-match
				       (and (eq (Varbdg-varname pair) var)
					    (varid= (Varbdg-vid pair) id-var)))
				    (p-match
				       (and (eq (Varbdg-skel pair) pat)
					    (varid= (Varbdg-skelid pair)
						    id-pat))))
				(:continue
				 :until (or v-match p-match)
				 :result (values (and v-match p-match)
						 pairings false))))))))))

   :occur-checker
      (\\ (var qvar-qexp vid qexp-id env)
	 (let ((b (uvar-lookup (Qexp-head qvar-qexp) qexp-id env)))
	    (cond (b
		   (let ((skel (Varbdg-skel b))
			 (skelid (Varbdg-skelid b)))
		      (let ((c (occur-check var skel vid skelid env)))
			 (cond ((eq c '*const)
				'*has-vars)
			       (t c)))))
		  ((and (varid= qexp-id vid)
			(eq (Qexp-head qvar-qexp)
			    var))
		   '*self)
		  (t '*has-vars))))

   :index-pat-maker
      (\\ (_) '*dontcare)

   :sexp-maker
     (\\ (qv) (make-Qvar (Qexp-head qv) !()))

   :main-op-extractor
      #'qvar-drill-for-main-op
)
)

;;; No longer used -- 
;;; 'var' is a Symbol, or symbol-like object (not a Qexp).
(defun try-bind-uvar (var val vid valid e)
   (declare (type Qvar var)
	    (type Qexp val)
	    (type Varid vid valid)
	    (type Bdgenv e))
   (let ((outcome
	  (occur-check var val vid valid e)))
      (case outcome
	 (*self (values true e))
	 ((*circular *has-boundvars) (values false e))
	 (t
	  (values true
		  (cons-Bdgenv
			 var vid
			 val (cond ((eq outcome '*has-vars) valid)
				   (t +ground-id+))
			 e))))))

(defun occur-check (var val vid valid env)
   (let ((occur-checker (Qexp-handler-occur-checker (Qexp-handler val))))
	 ;; The occur-checker can return several things:
         ;; *self -- the val is the same as the var, after looking through
	 ;; bindings.
	 ;; *circular -- the val properly contains the var
         ;; *has-boundvars -- contains occurrences of
         ;;     variables bound above here 
         ;; *const -- the val contains no vars at all, not even bound ones
	 ;; *has-vars -- the val contains vars, but not 'var'.
      (funcall occur-checker var val vid valid env)))

(defun maybe-not-occur-checked (pat)
   (cond ((and (atom pat) (not (is-Qvar pat)))
	  pat)
	 (t
	  (make-Not-occur-checked pat))))

(defun occur-check-res-combine (prev-res new-res)
   (ecase new-res
      ((*has-vars *circular *has-boundvars)
       new-res)
      ((*const)
       (cond ((eq prev-res '*const) '*const)
	     (t '*has-vars)))
      ((*self) '*circular)))

(defun list-occur-check (var l vid l-id env)
   (repeat :for ((x :in l)
		 (res '*const))
    :result res
    :within
      (let ((aoc (occur-check var x vid l-id env)))
	 (:continue
	  :until (memq aoc '(*self *circular))
	  :result '*circular
	  :until (eq aoc '*has-boundvars)
	  :result '*has-boundvars
	     (cond ((eq aoc '*has-vars)
		    (!= res '*has-vars)))))))

;;; Returns < success, pairings, discriminable >
;;;; 'discriminable' is discrim-bdgs* if pat1 and pat2 are not variants,
;;;; and they differ in having different constants at some position; otherwise,
;;;; 'discriminable' = false.
(defun qvariants (pat1 pat2 id1 id2 e)
   (qexp-variants pat1 pat2 id1 id2 (empty-env) e))

;;; 'pairings' is a Bdgenv used to pair qvars, so that skel is always
;;; a qvar name (*not* a Qvar).
;;; 'pairings' behaves like is a reversible a-list whose elements are Varbdgs
;;; Returns boolean + updated pairings + discriminable
;;; 'discriminable' is discrim-bdgs* if pat1 and pat2 are not variants,
;;; and they differ in having different constants at some position; otherwise,
;;; 'discriminable' = false.
(defun qexp-variants (pat1 pat2 id1 id2 pairings e)
   (declare (type Qexp pat1 pat2)
	    (type Varid id1 id2)
	    (type Bdgenv e))
   (let ((h1 (Qexp-handler pat1))
	 (h2 (Qexp-handler pat2)))
      (let ((p1 (Qexp-handler-upriority h1))
	    (p2 (Qexp-handler-upriority h2))
	    (v1 (Qexp-handler-variant-handler h1))
	    (v2 (Qexp-handler-variant-handler h2)))
	 (cond ((> p2 p1)
		(funcall v2 pat2 pat1 id2 id1 pairings e))
	       (t
		(funcall v1 pat1 pat2 id1 id2 pairings e))))))

(defun list-variants (l1 l2 id1 id2 pairings e)
   (repeat :for ((x1 :in l1)
		 (x2 :in l2))
    :result (values true pairings e)
    :within
       (multi-let (((ok pl d)
		    (qexp-variants x1 x2 id1 id2 pairings e)))
	  (:continue
	   :while ok
	   :result (values false pl d)
	      (!= pairings pl)))))

(defun place-qexp-index-pat (qexp)
   (let ((h (Qexp-handler qexp)))
      (cond ((Qexp-handler-long h)
	     ;; (We're assuming nil can't be the index-pat for
	     ;; a long qexp.)
	     (memoize-val
	        (funcall (Qexp-handler-index-pat-maker h)
			 qexp)
		:store-as (Long-Qexp-index-pat qexp)))
	    (t
	     ;; Don't bother caching it for small expressions
	     (funcall (Qexp-handler-index-pat-maker h)
		      qexp)))))

(defun qexp-main-op (qexp id env)
   (qexp-main-op-with-id qexp id id env))

(defun qexp-main-op-with-id (qexp id targ-id env)  
   (funcall (Qexp-handler-main-op-extractor (Qexp-handler qexp))
	    qexp id targ-id env))

;;; Creates a new variable with 'targ-id' and binds 'var-qexp' to it
;;; (with 'id').  Returns new env.  The effect is to replace unbound
;;; variable 'var-qexp' with a new unbound variable with the correct
;;; id.
(defun new-dummy-var (var-qexp id targ-id env)
		    (let ((new-var
			     (build-Qexp :qvar
				(gen-var 'dum)
				:type (Qexp-type var-qexp))))
		      (values new-var
			      (cons-Bdgenv
				 (Qexp-head var-qexp) new-var
				 id targ-id
				 env))))

;;; Follow var refs until we get a non-variable.  Return it + id.
;;; Compare with 'qnormalize' in qvarsubst.lisp, which returns a new
;;; env, leaving the id the same.
(defun qexp-normalize (q id env)
   (repeat
    :while (eq (Qexp-classname q) ':qvar)
    :within
       (let ((bdg (uvar-lookup (Qexp-head q) id env)))
	  (:continue
	   :while bdg
	      (!= q (Varbdg-skel bdg))
	      (!= id (Varbdg-skelid bdg))))
    :result (values q id)))

(defun qexps-equal (qe1 qe2)
   (let-fun ((:def sexps-search-for-ineq (e1 e2)
                (cond ((and (is-Qexp e1) (is-Qexp e2))
                       (qexps-equal e1 e2))
                      ((or (is-Qexp e1) (is-Qexp e2))
                       false)
                      ((and (atom e1) (atom e2))
                       (eq e1 e2))
                      ((or (atom e1) (atom e2))
                       false)
                      ((sexps-search-for-ineq
                          (car e1)
                          (car e2))
                       (sexps-search-for-ineq
                          (cdr e1)
                          (cdr e2)))
                      (t false))))
      (cond ((eq (Qexp-handler e1)
                 (Qexp-handler e2))
             (and (equal (Qexp-type e1)
                         (Qexp-type e2))
                  (sexps-search-for-ineq (Qexp-head e1)
                                         (Qexp-head e2))
                  (or (not (is-Long-Qexp e1))
                      (= (Long-Qexp-length e1)
                         (Long-Qexp-length e2))
                      (sexps-search-for-ineq (Qexp-args e1)
                                             (Qexp-args e2)))))
            (t
             false))))
