;-*- Mode: Common-lisp; Package: opt; Readtable: ytools; -*-
(in-package :opt)
;;; $Id: qvarsubst.lisp,v 1.15 2005/12/26 21:05:24 dvm Exp $

(depends-on %module/ ytools)

(depends-on :at-run-time %langutils/ symboid)

(depends-on :at-run-time %opt/ qunify)


;;; Utilities for cleaning up envs and formulas containing variables
;;; bound in those envs.

(eval-when (:compile-toplevel :load-toplevel :execute :slurp-toplevel)
   (export '(qexp-resolve safe-varsubst unsafe-varsubst varsubst-to-constant
	     env-elim bdgenvs-combine ids-env-elim
	     bdgenv-change-id varbdg-change-id varid-convert
	     env-simplify has-uvars)))

;;; Return a Qexp F equivalent to 'qexp' w.r.t. 'id' and 'env', such
;;; that all the variables in F's head and args are unbound in 'env'.
;;; This requires replacing all bound variables with their values,
;;; and renaming all unbound variables with id's other than 'id'.
(defun qexp-resolve (qexp id env)
   (cond ((or (varid= id +ground-id+)
              (cond ((Qexp-is-long qexp)
                     (null (Qexp-freevars qexp)))
                    (t
                     (not (memq (Qexp-classname qexp)
                                '(:qvar :boundvar))))))
	  ;; It's already variable-free
	  (values qexp env))
	 (t
	  (qexp-vars-elim qexp id id env))))

;;; Return resolvee + revised env (if revision is necessary)
(defun qexp-vars-elim (subexp sub-id targ-id env)
   (case (Qexp-classname subexp)
      ((:boundvar :qvar)
       (let ((varsym (Qexp-head subexp)))
	  (let ((bdg (uvar-lookup varsym sub-id env)))
	     (cond (bdg
		    (let ((val (Varbdg-skel bdg))
			  (val-id (Varbdg-skelid bdg)))
		       (cond ((eq (Qexp-classname val) ':boundvar)
			      (cond ((eq (Qexp-head val)
					 varsym)
				     ;; -- self-bind, set up by
				     ;; :binder code (below) in the usual way.
				     (values subexp env))
				    (t
				     (signal-problem qexp-vars-elim
					"Unexpected boundvar " val
					" as value of " subexp))))
			     (t
			      (qexp-vars-elim val val-id targ-id env)))))
		   ((varid= sub-id targ-id)
		    (values subexp env))
		   (t
		    (new-dummy-var subexp sub-id targ-id env))))))
      (:binder
       (multi-let (((body rest-or-key param-names)
		    (list->values (Long-Qexp-args subexp)))
		   ((_ rest-type param-types)
		    (list->values (Qexp-type subexp))))
	  (multi-let (((resolved-body env1)
		       (qexp-vars-elim
			  body sub-id targ-id 
			  (group-bind param-names
				      (<# (\\ (n ty)
					     (build-Qexp :boundvar
						n :type ty))
					  param-names
					  param-types)
				      sub-id sub-id
				      (cond ((is-Qexp rest-or-key)
					     (cons-Bdgenv
						(Qexp-head rest-or-key)
						(build-Qexp :boundvar
						   rest-or-key
                                                   :type rest-type)
						sub-id sub-id
						env))
					    (t env))))))
	     (values (build-Qexp :binder
			 (Qexp-head subexp)
			 :type (Qexp-type subexp)
			 :args (tuple resolved-body rest-type param-names))
		     env1))))
       (otherwise
	(cond ((Qexp-is-long subexp)
	       ;; In all remaining long cases, we can pretend that args are
	       ;; lists of Qexps, with possible exception of first arg
	       ;; for complex-apps and simple-key-apps
	       ;; Should beta-reduction be considered here?
	       (multi-let (((head env1)
			    (qexp-vars-elim (Qexp-head subexp)
					  sub-id targ-id env))
			   (ty (Qexp-type subexp)))
		  (let ((args (Long-Qexp-args subexp)))
		     (cond ((null args)
			    ;; simple app
			    (values (build-Qexp :simple-app
				       head
                                       :type ty
				       :args !())
				    env1))
			   (t
			    (multi-let (((resolved-args next-env)
					 (track-extra-vals
					    :extra ((next-env env1))
					    (<# (\\ (arg)
						   (extra-vals 
						      (qexp-vars-elim
							 arg
							 sub-id targ-id
							 next-env)
						      :+ next-env))
						(rest args)))))
			       (let-fun ()
				  (cond ((is-Qexp (first args))
					 ;; &rest arg
					 (multi-let (((rest-rsv env-z)
						      (qexp-vars-elim (first args)
								    sub-id sub-id
								    next-env)))
					    (rebuild (cons rest-rsv
							   resolved-args)
						     env-z)))
					(t
					 (rebuild (cons (first args)
							resolved-args)
						  next-env)))
				:where
				   (:def rebuild (resolved-args env-z)
					 (values
					    (make-Long-Qexp
					       :handler (Qexp-handler subexp)
					       :type (Qexp-type subexp)
					       :head head
					       :args resolved-args
					       :length (len resolved-args)
					       :freevars (args-freevars
							    resolved-args))
					    env-z)))))))))
	      (t
	       ;; Short Qexp, nothing to do
	       (values subexp env))))))

;;; This substitutes as far as possible, and as a consequence may have to
;;; return a revised env
(defun safe-varsubst (pat id alist)
   (cond (id
	  (subst-and-convert pat id id alist))
	 (t
	  (values pat alist))))

(defun subst-and-convert (pat subid destid alist)
   (cond ((is-Qvar pat)
          (cond ((eq (Qvar-sym pat) '_)
                 pat)
                (t
                 (let ((p (uvar-lookup (Qvar-sym pat) subid alist)))
                    (cond (p
                           (let ((val (Varbdg-skel p))
                                 (valid (Varbdg-skelid p)))
                              (subst-and-convert val valid destid alist)))
                          (t
                           (var-convert pat subid destid alist)))))))
	 ((atom pat)
	  (values pat alist))
	 (t
	  (multiple-value-bind (p1 a1)
			       (subst-and-convert
				  (car pat) subid destid alist)
	     (multiple-value-bind (p2 a2)
				  (subst-and-convert
				      (cdr pat) subid destid a1)
		(values (cons-if-new p1 p2 pat)
			a2))))))

;;; Return a version of 'env' in which all the bindings are of variables
;;; with given 'id'.
(defun env-simplify (env id)
   (cond ((not id) (empty-env))
	 (t
	  (repeat :for ((b :in env)
			(work-env env)
			:collector new-env)
	   :result (make-Bdgenv new-env)
	   :when (varid= (Varbdg-id b) id)
	   :within
	      (let ((val (Varbdg-skel b))
		    (valid (Varbdg-skelid b)))
		 (cond ((varid-is-ground valid)
			(:continue
			 :collect b))
		       (t
			(multi-let (((val new-work-env)
				     (exp-env-elim
				        val valid id
					work-env)))
			   (:continue
			       (!= work-env new-work-env)
			    :collect (make-Varbdg
				        (Varbdg-varname b) id
					val id))))))))))

; This is for debugging only, because it builds patterns with inconsistent
; varids.
; Actually, it is safe in the case where the value returned contains no
; vars.
(defun unsafe-varsubst (pat id alist)
   (cond ((is-Qvar pat)
	  (let ((p (uvar-lookup (Qvar-sym pat) id alist)))
	     (cond (p
		    (let ((val (Varbdg-skel p))
                          (valid (Varbdg-skelid p)))
		       (cond ((varid-is-ground valid) val)
			     (t
			      (unsafe-varsubst val valid alist)))))
		   (t pat))))
	 ((atom pat) pat)
	 (t
	  (let ((p1 (unsafe-varsubst (car pat) id alist))
		(p2 (unsafe-varsubst (cdr pat) id alist)))
	     (cons-if-new p1 p2 pat)))))

;;; Better
(defun varsubst-to-constant (pattern id alist)
   (letrec ((do-it (pat id alist)
	       (cond ((is-Qvar pat)
		      (let ((p (uvar-lookup (Qvar-sym pat) id alist)))
			 (cond (p
				(let ((val (Varbdg-skel p))
                                      (valid (Varbdg-skelid p)))
				   (cond ((varid-is-ground valid)
                                          val)
					 (t
					  (varsubst-to-constant
					      val valid alist)))))
			       (t
				(signal-problem varsubst-to-constant
				   "Variable " pat "[" id "] remaining in "
				    pattern
				      :% " after attempted varsubst"
				        " with [" id "] "
				      :% alist
				      (:continue
				       !"I'll proceed with variable where ~
                                         constant was expected"))))))
		     ((atom pat) pat)
		     (t
		      (let ((p1 (varsubst-to-constant (car pat) id alist))
			    (p2 (varsubst-to-constant (cdr pat) id alist)))
			 (cons-if-new p1 p2 pat))))))
      (do-it pattern id alist)))

;; Even less legal, but clearer.
(defun debug-varsubst (pat id alist)
   (cond ((is-Qvar pat)
	  (let ((p (uvar-lookup (Qvar-sym pat) id alist)))
	     (cond (p
		    (let ((val (Varbdg-skel p))
                          (valid (Varbdg-skelid p)))
		       (cond ((varid-is-ground valid)
                              val)
			     (t
			      (let ((r (debug-varsubst val valid alist)))
				 (cond ((has-qvars r)
                                        (vector r valid))
				       (t r)))))))
		   (t pat))))
	 ((atom pat) pat)
	 (t
	  (let ((p1 (debug-varsubst (car pat) id alist))
		(p2 (debug-varsubst (cdr pat) id alist)))
	     (cons-if-new p1 p2 pat)))))

;;; What does this do, in plain English?  (Maybe nothing interesting!)
(defun env-varid-convert (env source-id dest-id)
   (cond ((varid= source-id dest-id)
	  env)
	 (t
	  (!= env (list-copy *-*))
	  (repeat :for ((b :in env))
	     (let ((val (Varbdg-skel b))
		   (valid (Varbdg-skelid b)))
		(cond ((varid= valid source-id)
                       (!= < skel env >
                           (varid-convert skel source-id dest-id env))
                       (!= env
                           (cons (make-Varbdg
                                    (Varbdg-varname b)
                                    (Varbdg-id b)
                                    skel dest-id)
                                 (dremove1q b env)))))))
	  env)))

;;; We desire to combine x with another expression whose free vars
;;; have id dest-id.  To do so, we have to convert x from an expression
;;; whose free vars have source-id.  
;;; Returns new expression plus revised env
;;; x can be a list of expressions; the car is treated the same as
;;; other elements.
(defun varid-convert (x source-id dest-id env)
   (cond ((or (not source-id)
	      (varid= source-id +ground-id+)		; no free vars; trivial
	      (varid= source-id dest-id)
	      (env-is-empty env))
	  (values x env))
	 (t
	  (vars-id-convert x source-id dest-id env))))

;;; 'app' is a Long-Qexp of a class other than :binder.  
;;; Return < args, out-env >, where 'args' is a version of the 
;;; Long-Qexp-args of 'app' in which all free vars are bound
;;; with 'targ-id', and 'out-env' is a revised Bdgenv capturing
;;; the same info as 'env', but wrt 'targ-id'.
(defun simple-app-args-id-convert (app cur-id targ-id env)
   (let ((args (Long-Qexp-args app)))
      (cond ((or (varid= cur-id targ-id)
		 (varid= cur-id +ground-id+))
	     (values args env))
	    (t
	     ;; As usual, first arg gets special handling as rest-or-key
	     ;; spec.
	     (multi-let (((arg1 env1)
			  (cond ((is-Qexp (first args))
				 (qexp-convert-id (first args)
						  cur-id targ-id env))
				(t
				 (values (first args) env)))))
		(track-extra-vals :extra ((next-env env1))
		   (repeat :for ((a :in (rest args)))
		    :collect (extra-vals 
				(qexp-convert-id a cur-id targ-id next-env)
				:= next-env))))))))

(defun vars-id-convert (subval subid destid e)
   (cond ((or (not subid) (varid= subid destid))
	  (values subval e))
	 ((is-Qvar subval)
	  (let ((b (uvar-lookup (Qvar-sym subval) subid e)))
	     (cond (b
                    (let ((skel (Varbdg-skel b))
                          (id (Varbdg-skelid b)))
		       (vars-id-convert skel id destid e)))
		   (t
		    (var-convert subval subid destid e)))))
	 ((atom subval)
	  (values subval e))
	 (t
	  (do ((xl subval (cdr xl))
	       (x) 
	       (res '()))
	      ((null xl)
	       (values (nreverse res) e))
	      (setq x (car xl))
	      (multiple-value-bind
	                     (v newenv)
			     (vars-id-convert x subid destid e)
	         (setf res (cons v res))
		 (setq e newenv))))))

(defun var-convert (subval subid destid e)
  (cond ((varid= subid destid)
	 (values subval e))
	((> subid destid)
	 (let ((newsym (new-var-sym)))
	    (let ((newvar (make-Qvar newsym '())))
	       (let ((newenv
			(create-new-bdg 
			   (Qvar-sym subval)
			   newvar
			   subid destid
			   nil e)))
		   (values newvar newenv)))))
	(t
	 (let ((bb (uvar-lookup-backwards
		      (Qvar-sym subval)
		      subid destid e)))
	    (cond (bb
		   (values (make-Qvar
			      (Varbdg-varname
				 bb)
			      '())
			   e))
		  (t
		   (new-dummy-var
		      destid subval subid e)))))))

;;; Return a version of 'exp' in which no free variable has a binding
;;; in 'env.'  (So 'env' can be discarded if we don't care about
;;; any other variables it might bind.)
(defun env-elim (exp id env)
   (nth-value 0 (exp-env-elim exp id id env)))

;;; This generates envs that violate the prohibition on earlier varid's
;;; being bound to later ones, but it doesn't matter, because these
;;; envs will be discarded
(defun exp-env-elim (subexp subid destid alist)
   (cond ((is-Qvar subexp)
	  (let ((b (uvar-lookup (Qvar-sym subexp) subid alist)))
	     (cond (b
		    (let ((val (Varbdg-skel b))
                          (valid (Varbdg-skelid b)))
		       (exp-env-elim val valid destid alist)))
		   ((varid= subid destid)
		    (values subexp alist))
		   (t
		    (let ((newsym (new-var-sym)))
		       (let ((newvar (make-Qvar newsym '())))
			  (let ((newenv
				   (create-new-bdg 
				      (Qvar-sym subexp)
				      newvar
				      subid destid
				      nil alist)))
			      (values newvar newenv))))))))
	 ((atom subexp)
	  (values subexp alist))
	 (t
	  (multiple-value-bind (p1 a1)
			       (exp-env-elim
				  (car subexp) subid destid alist)
	     (multiple-value-bind (p2 a2)
				  (exp-env-elim
				      (cdr subexp) subid destid a1)
		(values (cons-if-new p1 p2 subexp)
			a2))))))

; Discard all bindings with the given id or later (greater)
;; Except optimist-vars, which must percolate all the way back to the top.
(defun bdgenv-contract (e discard)
   (cond ((null e) '())
         (t
          (let ((r (bdgenv-contract (cdr e) discard)))
             (cond ((and (>= (Varbdg-id (car e)) discard)
			 (or (not optimist-fn*)
			     (not (is-Optimist-var
				     (Varbdg-varname (car e))))))
                    r)
                   (t
                    (cons-if-new (car e) r e)))))))

(defun bdgenv-wrt-exp-simplify (env exp id)
   (bdgenv-simplify env (nodupq (raw-uvars exp)) id))

(defun bdgenv-simplify (env qvar-syms id)
   (let ((sought (<# (\\ (sym) (make-Varbdg sym id nil))
		     qvar-syms))
	 (tried !())
	 (new-bdgs !()))
      (repeat
       :until (null sought)
          (let ((var (Varbdg-varname (car sought)))
		(id (Varbdg-id (car sought))))
	     (!= tried (cons (car sought) *-*))
	     (!= sought (cdr *-*))
             (let ((b (uvar-lookup var id env)))
		(cond (b
		       (!= new-bdgs (cons b *-*))
		       (let ((skel (Varbdg-skel b))
                             (id (Varbdg-skelid b)))
			  (cond (id
                                 (let ((vl (raw-uvars skel)))
                                    (repeat :for ((v :in vl))
                                       (cond ((not (or (uvar-lookup
                                                          v id sought)
                                                       (uvar-lookup
                                                          v id tried)))
                                              (!= sought
                                                  (cons (make-Varbdg
                                                           v id nil)
                                                        *-*)))))))))))))
	:result (make-Bdgenv new-bdgs))))

; PAT may contain variables with DISCARD or higher.  Eliminate them,
; replacing them with new variables with id REPLACE instead.
(defun subst-away (pat e id discard replace)
   (cond ((is-Qvar pat)
          (let ((p (uvar-lookup (Qvar-sym pat) id e)))
             (cond (p
                    (let ((val (Varbdg-skel p))
                          (valid (Varbdg-skelid p)))
                       (cond ((varid-is-ground valid)
                              (values valid e))
                             (t
                              (subst-away val e valid
                                          discard replace)))))
                   ((varid= id replace)
                    (values pat e))
                   ((< id replace)
                    (multiple-value-bind (x e)
					 (new-dummy-var
                                            replace
                                            pat
                                            id
                                            e)
                       (values x e)))
                   (t
                    (let ((newvar (new-var-sym)))
                       (let ((e (cons (make-Varbdg
					    (Qvar-sym pat) id
                                            (make-Qvar newvar '())
                                            replace)
                                      (cons (make-Varbdg
                                                  newvar replace nil)
                                            e))))
                          (values (make-Qvar newvar '())
                                  e)))))))
         ((atom pat) (values pat e))
         (t
          (multiple-value-bind (a e1)
			       (subst-away (car pat) e id discard replace)
             (multiple-value-bind (d e2)
				  (subst-away (cdr pat) e1 id discard replace)
                (values (cons-if-new a d pat)
                        e2))))))

;; Simpler, faster interface to 'expression-car-cdr' for the usual case.
(defun e-car-cdr (exp id env)
   (cond ((and (consp exp) (not (is-Qvar exp)))
	  (values (car exp) (cdr exp) env))
	 (t
	  (multi-let (((_ a d e)
		       (expression-car-cdr exp id env true)))
	     (values a d e)))))

;;;; Get the car & cdr  of an expression.  In some cases, this may require
;;;; augmenting the environment, so return the new env.
;;;; If insist=false, then
;;;; first value returned is true if successful, false if exp is atomic
;;;; even after variable substitution.
;;;; Second value is atomic expression that results from substituting as
;;;; as possible.
;;;; If insist=true, then the same situation results in an error.
(defun expression-car-cdr (exp id env insist)
   (labels ((pursue (x subid)
	       (cond ((is-Qvar x)
		      (let ((p (uvar-lookup (Qvar-sym x) subid env)))
			 (cond (p
				(let ((val (Varbdg-skel p))
                                      (valid (Varbdg-skelid p)))
				   (pursue val valid)))
			       (insist
				(cerror
				   "Expression has no car and cdr ~s"
				   x))
			       (t
				(values false x nil env)
				))))
		     ((consp x)
		      (cond ((or (not subid) (varid= subid id))
			     (values true (car x) (cdr x) env))
			    (t
			     (multiple-value-bind
					(newx newenv)
					(varid-convert x subid id env)
				(values true (car newx) (cdr newx)
					newenv)))))
		     (insist
		      (error "Expression has no car and cdr ~s" x))
		     (t
		      (values false x nil env)))))
       (pursue exp id)))

;; Substitutes for mvars in pat, getting values from alist. 
; *Not guaranteed to substitute to any particular depth,* if
; going too far would cause it to insert variables with the wrong
; varid.
(defun varsubst (pat id alist)
   (cond ((null alist) pat)
	 (t (varsubst1 pat id alist))))

(defun varsubst1 (pat id alist)
   (cond ((is-Qvar pat)
	  (let ((p (uvar-lookup (Qvar-sym pat) id alist)))
	     (cond (p
		    (let ((val (Varbdg-skel p))
                          (valid (Varbdg-skelid p)))
		       (cond ((varid-is-ground valid)
                              val)
			     (t
                              (multiple-value-bind (x hasvars)
						   (pat-flatten
						       val valid alist)
                                 (cond (hasvars pat)
                                       (t x)))))))
		   (t pat))))
	 ((atom pat) pat)
	 (t
	  (let ((p1 (varsubst1 (car pat) id alist))
		(p2 (varsubst1 (cdr pat) id alist)))
	     (cons-if-new p1 p2 pat)))))

;; Try to merge env1 and env2, being careful about vars
;; bound in both.  Returns < ok, newenv >.  If envs are inconsistent,
;; first value is false.
(defun bdgenvs-combine (env1 env2)
   (cond ((null env1) (values true env2))
	 ((null env2) (values true env1))
	 (t
	  (let ((combined-env env1))
	     (repeat :for ((bdg2 :in env2)
			   ok skel skelid var id)
	      :result (values true combined-env)
	         (!= var (Varbdg-varname bdg2))
		 (!= id (Varbdg-id bdg2))
                 (!= skel (Varbdg-skel bdg2))
                 (!= skelid (Varbdg-skelid bdg2))
	         (!= < ok combined-env >
		     (var-unify var skel
				id skelid
				combined-env))
	       :while ok
	       :else result (values false nil))))))
				
;;; Eliminate all references to the given ids from env.
;;; All elements of 'ids' are Varid's (not false).
(defun ids-env-elim (ids env)
   (let-fun ((:def id-is-bad (id) (member id ids))

             (:def bdg-is-bad (b)
	       (cond ((id-is-bad (Varbdg-id b))
		      ;; We don't care about bindings of bad vars
		      false)
		     (t
		      ;; We do care about bindings of good vars to bad
		      (let ((val-id (Varbdg-skelid b)))
			    (and (not (varid-is-ground val-id))
                                 (id-is-bad val-id))))))

             (:def bdg-has-bad-id (b)
	       (or (id-is-bad (Varbdg-id b))
		   (bdg-is-bad b))))
;;;;      (out (:i> 3)
;;;;	   "env = "
;;;;	   (:e (<# (\\ (b)
;;;;		      (:o "[" b "/" (:q ((bdg-has-bad-id b)
;;;;					 "bad")
;;;;					(t "good"))
;;;;			  "]" :%))
;;;;		   env))
;;;;	   :%)
      (cond ((and (not (null ids))
		  (exists (b :in env)
		     (bdg-has-bad-id b)))
	     (let ((var-repl-tab !((Lrcd Symbol Symbol))))
	            ;; -- replacements for all vars with bad ids
                (let ((repl-id
		         (</ (\\ (low-id b)
				(cond ((bdg-is-bad b)
				       (cond (low-id
					      (min low-id (Varbdg-id b)))
					     (t (Varbdg-id b))))
				      (t low-id)))
			     false env)))
		   (repeat :for ((bdgs-to-check = env :then check-bdgs)
				 (check-bdgs = !() :then :again)
				 (good-bdgs = !())
				 (curenv = env
					 :then (append check-bdgs curenv)))
		      (repeat :for ((b :in bdgs-to-check))
			 (cond ((bdg-is-bad b)
				(let ((val (Varbdg-skel b))
                                      (val-id (Varbdg-skelid b)))
				   (multi-let (((newskel new-curenv)
                                                (safe-varsubst
                                                    val val-id curenv)))
                                      (!= check-bdgs
                                          (nconc (ldiff new-curenv curenv)
                                                 *-*))
                                      (!= curenv new-curenv)
                                      (!= good-bdgs
                                          (cons (make-Varbdg
                                                   (Varbdg-varname b)
                                                   (Varbdg-id b)
                                                   (skel-subst
                                                      newskel))
                                                *-*)))))))
;;;;		      (out "bdgs-to-check = "
;;;;			   (:e (<# (\\ (b)
;;;;				      (:o "[" b "/" (:q ((bdg-has-bad-id b)
;;;;							 "bad")
;;;;							(t "good"))
;;;;					  "]" :%))
;;;;				   env))
;;;;			   :% "check-bdgs = " check-bdgs
;;;;			   :% "good-bdgs = " good-bdgs
;;;;			   :%)
		    :until (null check-bdgs)
		    :result (nconc good-bdgs
				   (<? (\\ (b)
					  (not (bdg-has-bad-id b)))
				       curenv))
		    :where
		      (skel-subst (skel)
			 (let ((found-var false))
			    (letrec ((do-subst (x)
					(cond ((is-Qvar x)
					       (!= found-var true)
					       (var-replace x))
					      ((atom x) x)
					      (t
					       (cons-if-new
						  (do-subst (car x))
						  (do-subst (cdr x))
						  x)))))
			       (let ((newskel (do-subst skel)))
				  (make-Expclo
				     newskel
				     (cond (found-var repl-id)
					   (t false)))))))
		      (var-replace (qvar)
			 (let ((p (assq (Qvar-sym qvar) var-repl-tab)))
			    (cond ((not p)
				   (!= p (tuple (Qvar-sym qvar)
						(new-var-sym)))
				   (!= var-repl-tab
				       (cons p *-*))))
			    (make-Qvar (cadr p) !())))))))
	    (t env))))

(defun bdgenv-change-id (env old-id new-id)
   (cond ((or (varid= old-id +ground-id+)
              (varid= new-id +ground-id+))
          (cond ((or (null env)
                     (and (varid= old-id +ground-id+)
                          (varid= new-id +ground-id+)))
                 env)
                (t
                 (signal-problem bdgenv-change-id
                    "Incoherent call to bdgenv-change-id:"
                    :% "  old-id = " old-id "  new-id = " new-id
                    :% "  env = " env))))
         ((varid= old-id new-id)
          env)
         (t
          (make-Bdgenv
             (<# (\\ (b)
                    (varbdg-change-id b old-id new-id))
                 (Bdgenv-bdgs env))))))

(defun varbdg-change-id (b oldid newid)
   (make-Varbdg
      (Varbdg-varname b)
      (cond ((varid= (Varbdg-id b) oldid)
	     newid)
	    (t (Varbdg-id b)))
      (let ((val (Varbdg-val b)))
	 (let ((skel (Expclo-skel val))
	       (skelid (Expclo-id val)))
	    (cond ((or (not skelid)
		       (not (varid= skelid oldid)))
		   val)
		  (t
		   (make-Expclo skel newid)))))))

;;; Substitute until expression is an unbound variable if it is a variable
;;; at all.  Augment env if 
(defun qnormalize (qexp required-id env)
   (let-fun ((:def pursue (subexp id)
		(cond ((eq (Qexp-classname subexp) ':qvar)
		       (let ((bdg (uvar-lookup (Qexp-head subexp) id env)))
			  (cond (bdg
				 (pursue (Varbdg-skel bdg)
					 (Varbdg-skelid bdg)))
				(t
				 (got subexp id)))))
		      (t
		       (got subexp id))))

	     (:def got (subexp id)
		(cond ((or (varid= id required-id)
                           (varid= id +ground-id+))
		       (values subexp env))
		      (t
		       (bind-new-var id subexp))))

	     (:def bind-new-var (id subexp)
	      ;; We've found unpursuable var, but with the wrong id, so we
	      ;; need to bind the old one to a new var with the right one --
		(let ((new-var-name (gen-var 'qv)))
		   (let ((new-var-qexp
			    (build-Qexp :qvar
			       new-var-name
			       :type (Qexp-type subexp))))
		      (values new-var-qexp
			      (cons-Bdgenv
				 (Qexp-head subexp) id new-var-qexp required-id
				 env))))))
      (pursue qexp required-id)))

;;; Equivalent --
;;;;   (repeat
;;;;    :while (eq (Qexp-classname qexp) ':qvar)
;;;;    :within (let (bdg (uvar-loookup (Qexp-head qexp) id env))
;;;;	       (:continue
;;;;		:while bdg
;;;;		   (!= qexp (Varbdg-skel bdg))
;;;;		   (!= id (Varbdg-id bdg))))
;;;;    :result (value qexp id))

;;; Are any variables in qexp still unassigned?  (Are there any
;;; unbound freevars except those in 'okay-vars'?)
(defun has-uvars (qexp okay-vars id env)
   (let-fun ((:def search-down (qexp id)
		(cond ((Qexp-is-long qexp)
		       (exists (fv :in (Long-Qexp-freevars qexp))
			  (cond ((memq fv okay-vars)
				 false)
				(t
				 (check-val fv id)))))
		      ((eq (Qexp-classname qexp) ':qvar)
		       (check-val (Qexp-head qexp) id))
		      (t false)))
	     (:def check-val (fv id)
		(let ((b (uvar-lookup fv id env)))
		   (cond (b
			  (search-down (Varbdg-skel b)
				       (Varbdg-skelid b)))
			 (t true)))))
      (search-down qexp id)))