;-*- Mode: Common-lisp; Package: opt; Readtable: ytools; -*-
(in-package :opt)

(depends-on %module/ opt)

#|
What's needed to make this a useful tool --

Change parameters of 'resolve' so it takes two args, the second optional.
Each arg is of the form (clause literal-position).  However, the clause 
and position can be switched, and either can be omitted, in which case
clause defaults to the last one generated, and literal-position
defaults to 0.  When one is omitted, the parens can be omitted too.
(Other ideas: specifying literal by its predicate instead of
position; or allowing both.)

If literal-position is missing and the zero interpretation doesn't
work, the system should figure out which literals resolve and display
all possible resolutions.

The current hack is totally global. There's just one workspace, the
global variables whose values are the clauses.  The list all-clauses*
is supposed to be an alist of (var-name var-val) pairs.  See
TEACH/ai/05f/assignments/midterm.lisp. 
There should instead be a "current clause set," and it should be
possible to switch among alternative sets.  Lemma management.

Each clause should have a proof attached.  The only proof now is the
record of inferring the empty clause, which includes false steps.

|#


(defvar last-clause-var*)

(defmacro resolve (&key ((:c1 c1^) last-clause-var*)
                        ((:p1 pos1^) 0)
                        ((:c2 c2^) last-clause-var*)
                        ((:p2 pos2^) 0))
   (let ((new-clause-var (build-symbol clause (++ clause-counter*))))
      `(progn (defvar ,new-clause-var)
              (setq ,new-clause-var
                    (do-resolve ,c1^ ,pos1^ ,c2^ ,pos2^))
              (!= last-clause-var* ',new-clause-var)
              (!= (alref all-clauses* ',new-clause-var)
                  ,new-clause-var)
              (tuple ',new-clause-var ,new-clause-var))))

;;; Factors if the two clauses are the same
(defun do-resolve (c1 p1 c2 p2)
   (let* ((dl1 (clause-disjuncts c1))
          (dl2 (clause-disjuncts c2))
          (id1 (new-Varid))
          (id2 (cond ((eq c1 c2) id1)
                     (t (new-Varid)))))
      (let-fun ()
         (!= p1 (pos-in *-* dl1))
         (!= p2 (pos-in *-* dl2))
         (let ((lit1 (nth p1 dl1))
               (lit2 (nth p2 dl2)))
            (multi-let (((ok theta)
                         (cond ((eq c1 c2)
                                (unify lit1 lit2 id1 id1 (empty-env)))
                               (t
                                (or (cancels lit1 lit2 id1 id2)
                                    (cancels lit2 lit1 id1 id2))))))
	       (cond (ok
                      (multi-let (((new-clause-maybe-dups env)
                                   (cond ((eq dl1 dl2)
                                          (values
                                             (remove-from-two-pos)
                                             theta))
                                         (t
                                          (union-after-cancel env)))))
                         ;; Now remove duplicates
                         (do ((rem new-clause-maybe-dups (rest rem))
                              (new-dl
                                  '()
                                  (cond ((member (car rem) (cdr rem)
                                                 :test #'equalp)
                                         new-dl)
                                        (t (cons (car rem) new-dl)))))

                             ((null rem)
                              (new-clause new-dl)))))
                     (t
                      (signal-problem do-resolve
                         "Clauses do not resolve on literals "
                         p1 " and " p2 ":"
                         :% 3 c1 :% 3 c2)))))
         :where

           ;; Interpret 'e' as a position in 'l', in the usual way
           ;; if non-negative; otherwise counting from the other end.
           (:def pos-in (e l)
               (cond ((< e 0)
                      (+ (length l) e))
                     (t e)))

           (:def cancels (l m id-l id-m)
              (and (consp l)
                   (eq (car l) 'not)
                   (unify (cadr l) m id-l id-m (empty-env))))

           (:def union-after-cancel (env)
              (track-extra-vals :extra ((env-x env))
                 (let ((dl1-subst
                           (extra-vals (env-a)
                              (safe-varsubst
                                 (append (drop (+ p1 1) dl1)
                                         (take p1 dl1))
                                 id1 env-x)
                              :+ (env-x env-a)))
                       (dl2-subst
                           (extra-vals (env-b)
                              (varid-convert dl2 id2 id1 env)
                              :+ (env-x env-b))))
                    (

              (control-nest
               :subst1
                 (multi-let (((dl1-subst env-a)
                              (safe-varsubst 
                                 (append (drop (+ p1 1) dl1)
                                         (take p1 dl1))
                                 id1 env)))
                    :subst2)
               :subst2
                 (multi-let (((dl2-subst env-b)
                              (varid-convert dl2 id2 id1 env)))
                    :
                 (values
                    (append (drop (+ p2 1) dl2-a-la-1)
                            (take p2 dl2-a-la-1)
                            (drop (+ p1 1) dl1)
                            (take p1 dl1))
                    env-2-1)))

           (:def combine-two-lits (env)
              (cond ((= p1 p2)
                     ;; Degenerate case
                     dl1)
                    (t
                     (let ((q (max p1 p2)))
                        (append (take q dl1)
                                (drop (+ q 1) dl1)))))))
                                 (ldiff (tail tail1)
                                        tail2)
                                 (tail tail2))))
                  id1 env))))))

(defun rename-variables (x)
  "Replace all variables in x with new ones."
   (let ((subst-tab (mapcar #'(lambda (var) (make-Bdg var (new-variable var) t))
                            (variables-in x))))
      (substitution-apply subst-tab x)))

(defun substitution-apply (subst-tab x)
      (let-fun ()
         (subst-vars x)
       :where
          (:def subst-vars (subexp)
              (cond ((is-Qvar subexp)
                     (get-binding-val (Qvar-sym subexp) subst-tab subexp))
                    ((atom subexp)
                     subexp)
                    (t
                     (<# subst-vars subexp))))))

;;; Assume it's already a disjunction, with 'or' omittable if it has
;;; exactly one disjunct --
(defun clause-disjuncts (c)
   (cond ((car-eq c 'or) (cdr c))
         (t (list c))))

(defun new-clause (literals)
   (cond ((= (length literals) 1)
          (first literals))
         (t
          `(or ,@literals))))