;-*- Mode: Common-lisp; Package: opt; Readtable: ytools; -*-
(in-package :opt)
;;; $Id: unify.lisp,v 1.14 2004/12/15 22:38:02 dvm Exp $

(depends-on %module/ ytools)

(depends-on :at-run-time %langutils/ symboid)

;; If not false, the function that triggers optimistic matching.
;; Used in svcplnr and Optop, where optimist-fn* = step-value.
(in-header 
   (defvar optimist-fn* false))

(eval-when-slurping
   (defvar discrim-bdgs*))

(end-header)

(declaim (optimize (safety 1) (speed 2)))

;;; This unifier renames variables in constant time by defining a
;;; variable as a pair <symbol, varid>, where varid is an arbitrary integer.
;;; To rename all the variables in an expression, just generate a new varid.
;;; For technical reasons, it's important that no variable with varid I is
;;; bound to a variable with varid J>I.  Varids grow monotonically, so
;;; this means that no "old" variable is bound to a "new" variable.  

(eval-when (:compile-toplevel :load-toplevel :execute :slurp-toplevel)
   (export '(unify variants)))

(defvar vid* 0)

(defun new-Varid () (setq vid* (+ vid* 1)) vid*)

;;; Id to use when you're sure the pattern has no variables
(defvar ground-id* false)

;;; Useful when you need a temporary id, and 
;;; you're sure the other pat isn't using the dummy already.
(defvar dummy-id* (new-Varid))
(defvar dummy-id2* (new-Varid))

(defstruct (Expclo (:type vector)
		   (:constructor make-Expclo (skel id)))
   skel
   (id nil)) ;  :type integer or nil

;Key idea: New data type, bdgenv, which is a list of entries of the form:
;   (varname varid val)
;which specifies the value val of variable named varname with id varid

(defstruct (Varbdg (:type list)
		   (:constructor make-Varbdg (varname id val)))
   (varname nil) ;; :type Symbol or Optimist-var
   (id nil :type integer)
   val)

(def-class Optimist-var
   id   ; Integer
   (:handler
      (print (ov srm)
	 (out (:to srm) "(-:" (Optimist-var-id ov) ":-)"))))

(defvar ov-count* 0)

(defun new-Optimist-var ()
   (!= ov-count* (+ *-* 1))
   (make-Optimist-var ov-count*))

;; Records instance of two things unifying that a realist would
;; refuse to unify.  Or vice versa if polarity = false.
;; opt-pat is the pattern whose function is optimist-fn*.
;; An Optimism-rcd is usually paired with another occurrence
;; of other-pat, so it will get properly id-ified.
(def-class Optimism-rcd
   polarity opt-pat other-pat id1 id2)

(declaim (inline vars=))

(defun vars= (v1 v2 i1 i2)
   (and (eq v1 v2) (eql i1 i2)))

(defun make-Bdgenv (bdgs) bdgs)

(defun Bdgenv-bdgs (e) e)

(defun empty-env () '())

(defun env-is-empty (env) (null env))

(defvar remove-duplicate-bdgs* t)

(defun cons-Bdgenv (b prev e)
  (cons b 
	(cond ((and prev remove-duplicate-bdgs*)
	       (remove prev e :count 1 :test #'eq))
	      (t e))))

(defun uvar-lookup (name id env)
   (do     ((bl env (cdr bl)))
	   ((or (null bl)
		(vars= (Varbdg-varname (car bl))
		       name
		       (Varbdg-id (car bl))
		       id))
	    (cond ((null bl) nil)
		  (t (car bl))))))

;;; Version that ignores ids.  (Used only for hackish purposes in 'deduction')
(defun uvar-find (name env)
   (assq name env))

; Find a binding of a variable with eid in env whose value
; is variable with given name and id.  (id is never nil)
(defun uvar-lookup-backwards (name id eid env)
   (do ((bl env (cdr bl)))
       ((or (null bl)
	    (and (= (Varbdg-id (car bl))
		    eid)
		 (let ((v (Varbdg-val (car bl))))
		    (let ((vsk (Expclo-skel v))
			  (vid (Expclo-id v)))
		       (and vid
			    (= vid id)
			    (is-Qvar vsk)
			    (eq (Qvar-sym vsk)
				name))))))
	(cond ((null bl) nil)
	      (t (car bl))))))

(declaim (inline no-new-bdgs first-new-bdg))

(defun no-new-bdgs (new old)
   (eq new old))

(defun first-new-bdg (new old)
   (declare (ignore old))
   (car new))

(defun new-bdgs (new old)
   (ldiff new old))

(defun first-real-bdg (new old)
   (loop
      (cond ((or (null new) (eq new old))
	     (error "No real bdgs: ~s~%  [before ~s]" new old))
	    ((Varbdg-val (car new))
	     (return (car new))))
      (setq new (cdr new))))

(defvar unify-count* 0)
(defvar unify-success-count* 0)

; t1 and t2 are terms, e1 and e2 are varids, e is bdgenv
(defun unify (t1 t2 e1 e2 e)
   (setq unify-count* (+ unify-count* 1))
   (multiple-value-bind (ok e)
			(pat-unify t1 t2 e1 e2 e)
      (cond (ok (setf unify-success-count* (+ unify-success-count* 1))))
      (values ok e)))

; Not really a bdgenv; used as flag to tell caller how match failed.
(defvar discrim-bdgs* (list (make-Varbdg 'discrim 0
					 (make-Expclo 'discrim 'nil))))

(defun pat-unify (t1 t2 e1 e2 e)
   (cond ((is-Qvar t1)
          (var-unify (Qvar-sym t1) t2 e1 e2 e))
         ((is-Qvar t2)
          (var-unify (Qvar-sym t2) t1 e2 e1 e))
         ((or (atom t1) (atom t2))
	  (strip-sym-with-type t1)
	  (strip-sym-with-type t2)
          (cond ((equal t1 t2)
                 (values t e))
		(optimist-fn*
		 (multi-let (((_ a1 a2)
			      (eq-stripping-sym-with-type
			         (and (consp t1) (car t1))
				 (and (consp t2) (car t2)))))
		    (optimist-unify a1 a2 t1 t2 e1 e2 e)))
                (t
                 (values nil discrim-bdgs*))))
         (t
	  (multi-let (((same a1 a2)
		       (eq-stripping-sym-with-type (car t1) (car t2))))
	     (cond (same
		    (list-unify (cdr t1) (cdr t2) e1 e2 e))
		   (optimist-fn*
		    (optimist-unify a1 a2 t1 t2 e1 e2 e))
		   (t (values nil discrim-bdgs*)))))))

(defun list-unify (t1l t2l e1 e2 e)
          (do ((l1 t1l (cdr l1))
               (l2 t2l (cdr l2))
	       (okaysofar t))
              ((or (null l1) (null l2)
                   (not okaysofar))
	       (values (and okaysofar (null l1) (null l2))
                       e))
	    (multiple-value-setq (okaysofar e)
				 (pat-unify (car l1) (car l2) e1 e2 e))))

;; car1 and car2 are the cars of t1 and t2, with Sym-with-type's stripped
;; If t1 or t2 is atomic, the corresponding car is false.
(defun optimist-unify (car1 car2 t1 t2 id1 id2 e)
   (letrec ((env-aug-with-optimist-bdg (ta tb ida idb)
	       (values t
		       (create-new-bdg
			  (new-Optimist-var)
			  (list (make-Optimism-rcd
				   true ta tb ida idb)
				tb)
			  idb idb false e))))
      (cond ((eq car1 optimist-fn*)
	     (env-aug-with-optimist-bdg t1 t2 id1 id2))
	    ((eq car2 optimist-fn*)
	     (env-aug-with-optimist-bdg t2 t1 id2 id1))
	    (t (values nil discrim-bdgs*)))))

(defun var-unify (var val varid valid e)
   (cond ((eq var '_)
	  (values true e))
	 (t
	  (let ((bdg (uvar-lookup var varid e)))
	     (cond ((and bdg (Varbdg-val bdg))
		    (old-var-unify bdg val valid e))
		   (t
		    (try-bind-uvar var val varid valid e)))))))

(defun old-var-unify (bdg val valid e)
             (let ((varval (Varbdg-val bdg)))
                (multiple-value-bind (ok sub-e)
				     (pat-unify (Expclo-skel varval) val
		                                (Expclo-id varval)   valid
						e)
                   (cond (ok (values ok sub-e))
                         (t (values nil e))))))

(defun try-bind-uvar (var val varid valid e)
   (let ((destid (and valid (min varid valid))))
      (labels ((resolve-vars (subval subid e)
                  (cond ((not subid)
			 (values '*const subval e))
			((< subid varid)
			 (values '*var subval e))
			((is-Qvar subval)
			 (let ((b (uvar-lookup (Qvar-sym subval) subid e)))
			   (cond ((and b (Varbdg-val b))
				  (let ((v (Varbdg-val b)))
				    (let ((skel (Expclo-skel v))
					  (id (Expclo-id v)))
				      (multiple-value-bind
						(c v e)
						(resolve-vars skel id e)
					 (cond ((eq c '*var)
						(cond ((= id destid)
						       (values '*var v e))
						      ((= subid destid)
						       (values '*var subval e))
						      (t
					               ; subid > destid
						       (flag-new-dummy-var
                                                        destid v
                                                        (min id destid)
                                                        e))))
					       (t (values c v e)))))))
				 ((= subid destid)
				  (values (cond ((eq (Qvar-sym subval) var)
						 '*self)
						(t '*var))
					  subval e))
				 (t
				  ; SUBID > DESTID
				  (let ((newsym (new-var-sym)))
				    (let ((newvar (make-Qvar newsym '())))
				       (let ((newenv
					        (create-new-bdg 
					           (Qvar-sym subval)
						   newvar
						   subid destid
						   b e)))
					 (values '*var
						 newvar
						 newenv))))))))
			((atom subval)
			 (values '*const subval e))
			(t
			 (do ((xl subval (cdr xl))
			      (x) 
                              (res '())
                              (v) (c)
                              (outcome '*const))
			     ((null xl)
			      (values outcome (nreverse res) e))
			   (setq x (car xl))
			   (multiple-value-setq (c v e)
						(resolve-vars x subid e))
			   (cond ((member c '(*self *circular) :test #'eq)
				  (return (values '*circular nil e))))
			   (setf res (cons v res))
			   (cond ((eq c '*var)
				  (setf outcome '*var))))))))
         (multiple-value-bind (outcome newval newenv)
			      (resolve-vars val valid e)
            (case outcome
               (*self (values t e))
               (*circular (values nil e))
               (t
		(values t
			(create-new-bdg
			       var newval
                               varid (cond ((eq outcome '*var) destid)
                                           (t nil))
                               nil
			       newenv))))))))

(defun flag-new-dummy-var (destid valskel valid e)
   (multiple-value-bind (newvar newenv)
			(new-dummy-var destid valskel valid e)
      (values '*var newvar newenv)))

(defun new-dummy-var (destid valskel valid e)
   (let ((newvar (new-var-sym)))
      (values (make-Qvar newvar '())
	      (create-new-bdg newvar valskel destid valid nil e))))

;;;;              (cons-Bdgenv (make-Varbdg newvar destid
;;;;                                    (make-Expclo valskel valid))
;;;;                       nil
;;;;                       e))))

(defun variants (pat1 pat2 id1 id2 e)
   (pat-variants pat1 pat2 id1 id2 '() '() e))

;;;; m1 and m2 are lists of expclos.  Elt i of m1 corresponds to elt i of m2,
;;;; and vice versa.
;;;; Returns boolean + updated m1 + updated m2 + discriminable
;;;; 'discriminable' is discrim-bdgs* if pat1 and pat2 are not variants,
;;;; and they differ in having different constants at some position; otherwise,
;;;; 'discriminable' = false.
(defun pat-variants (pat1 pat2 id1 id2 m1 m2 e)
   (cond ((is-Qvar pat1)
	  (var-variant pat1 pat2 id1 id2 m1 m2 e))
	 ((is-Qvar pat2)
	  (var-variant pat2 pat1 id2 id1 m2 m1 e))
	 ((or (atom pat1) (atom pat2))
	  (strip-sym-with-type pat1)
	  (strip-sym-with-type pat2)
	  (cond ((eql pat1 pat2)
		 (values true m1 m2 nil))
		(t
		 (values false m1 m2 discrim-bdgs*))))
	 (t
	  (multi-let (((same _ _)
		       (eq-stripping-sym-with-type (car pat1) (car pat2))))
	     (cond (same
		    (do ((l1 (cdr pat1) (cdr l1))
			 (l2 (cdr pat2) (cdr l2))
			 (okaysofar t) d)
			((or (null l1) (null l2)
			     (not okaysofar))
			 (cond ((and okaysofar (null l1) (null l2))
				(values true m1 m2 nil))
			       (t
				(values false m1 m2
					(cond (okaysofar discrim-bdgs*)
					      (t d))))))
		      (multiple-value-setq (okaysofar m1 m2 d)
					   (pat-variants (car l1) (car l2)
							 id1 id2 m1 m2 e))))
		   (t (values false m1 m2 discrim-bdgs*)))))))

(defun var-variant (var pat id-var id-pat m-var m-pat e)
   (let ((bdg (uvar-lookup (Qvar-sym var)
			   id-var e)))
      (cond ((and bdg (Varbdg-val bdg))
             (let ((varval (Varbdg-val bdg)))
	        (pat-variants (Expclo-skel varval)
			      pat
			      (Expclo-id varval)
			      id-pat
			      m-var m-pat e)))
	    (t
	     (loop
	        (cond ((is-Qvar pat)
		       (let ((pb (uvar-lookup (Qvar-sym pat)
					      id-pat e)))
			  (cond ((and pb (Varbdg-val pb))
				 (let ((patval (Varbdg-val pb)))
				    (setq pat (Expclo-skel patval))
				    (setq id-pat (Expclo-id patval))))
				(t
				 (return nil)))))
		      (t (return nil))))
	     (cond ((is-Qvar pat)
		    (do ((mvl m-var (cdr mvl))
			 (mpl m-pat (cdr mpl))
			 (vname (Qvar-sym var))
			 (pname (Qvar-sym pat))
			 v-match p-match)
			((or (null mvl)
			     (progn
			         (setq v-match
				       (and (eq (Expclo-skel (car mvl))
						vname)
					    (= (Expclo-id (car mvl))
					       id-var)))
				 (setq p-match
				       (and (eq (Expclo-skel (car mpl))
						pname)
					    (= (Expclo-id (car mpl))
					       id-pat)))
				 (or v-match p-match)))
			 (cond ((null mvl)
				(values true
					(cons (make-Expclo vname id-var)
					      m-var)
					(cons (make-Expclo pname id-pat)
					      m-pat)
					false))
			       ((and v-match p-match)
				(values true m-var m-pat nil))
			       (t
				(values false m-var m-pat false))))))
		   (t
		    (values false m-var m-pat discrim-bdgs*)))))))

(defun eq-stripping-sym-with-type (a1 a2)
   (strip-sym-with-type a1)
   (strip-sym-with-type a2)
   (values (eq a1 a2) a1 a2))

; no longer used:
(defun pat-flatten (x id alist)
   (cond ((is-Qvar x)
          (let ((p (uvar-lookup (Qvar-sym x) id alist)))
             (cond ((and p (Varbdg-val p))
                    (let ((val (Varbdg-val p)))
                       (cond ((Expclo-id val)
                              (multiple-value-bind (y hasvars)
						   (pat-flatten
						      (Expclo-skel val)
						      (Expclo-id val)
						      alist)
                                 (cond (hasvars (values nil t))
                                       (t (values y nil)))))
                             (t (values (Expclo-skel val) nil)))))
                   (t (values nil t)))))
         ((atom x) (values x nil))
         (t
          (multiple-value-bind (p1 v1)
			       (pat-flatten (car x) id alist)
             (cond (v1 (values nil t))
                   (t
                    (multiple-value-bind (p2 v2)
                                         (pat-flatten (cdr x) id alist)
                       (cond (v2 (values nil t))
                             (t (values (cons-if-new p1 p2 x) nil))))))))))


(defun create-new-bdg (var val varid valid bdg e)
      (cons-Bdgenv (make-Varbdg var varid
			    (make-Expclo val valid))
	       bdg e))

(defun has-qvars (x)
   (cond ((is-Qvar x) t)
	 ((atom x) nil)
	 (t (some #'has-qvars x))))

(defun has-uvars (x id e)
   (cond ((is-Qvar x)
	  (let ((b (uvar-lookup (Qvar-sym x) id e)))
	     (cond (b
		    (let ((ec (Varbdg-val b)))
		       (cond ((Expclo-id ec)
			      (has-uvars (Expclo-skel ec)
					 (Expclo-id ec)
					 e))
			     (t nil))))
		   (t t))))
	 ((atom x) nil)
	 (t
	  (some #'(lambda (y) (has-uvars y id e))
		x))))

(defun all-uvars (x id e)
   (cond ((is-Qvar x)
          (let ((b (uvar-lookup (Qvar-sym x) id e)))
             (cond ((and b (Varbdg-val b))
		    (let ((ec (Varbdg-val b)))
                       (all-uvars (Expclo-skel ec)
				  (Expclo-id ec)
				  e)))
                   (t
                    (list (make-Expclo x id))))))
         ((atom x) '())
         (t
	  (mapcan #'(lambda (y) (all-uvars y id e))
		  x))))

; Useful for processing input formulas, with no binding history yet
(defun raw-uvars (x)
   (cond ((is-Qvar x) (list (Qvar-sym x)))
	 ((atom x) '())
	 (t
	  (reduce #'(lambda (coll r)
		       (union coll (raw-uvars r) :test #'eq))
	          x
		  :initial-value '()))))

(defun same-expclo-var (ev1 ev2)
   (let ((v1 (Expclo-skel ev1))
	 (v2 (Expclo-skel ev2))
	 (id1 (Expclo-id ev1))
	 (id2 (Expclo-id ev2)))
      (and (eq (Qvar-sym v1) (Qvar-sym v2))
	   (cond (id1 (and id2 (= id1 id2)))
		 (t (null id2))))))

(defvar varno* 0)

(defun new-var-sym ()
   (setq varno* (+ varno* 1))
   (build-symboid _ (< (let ((*print-radix* nil))
			  (princ-to-string varno*)))))
;   (intern
;	      (concatenate 'string
;		 "_" 
;		 )))

(defun one-bindings-subst (bdgs-set id allow-others)
   (cond ((null bdgs-set)
	  (error "Deductive answer expected and none found"))
	 (t
	  (let ((a (bindings-subst (car bdgs-set) id)))
	     (cond ((and (not allow-others)
			 (not (null (cdr bdgs-set))))
		    (cerror "I will ignore all but one"
			    "Unexpected multiple answers to deductive query ~s"
			    bdgs-set)))
	     a))))

(defun bindings-subst (bdgs id)
   (let (v x i (res '()))
      (loop for b in bdgs
	 do (cond ((= (Varbdg-id b) id)
		   (setq v (Varbdg-val b))
		   (setq x (Expclo-skel v))
		   (setq i (Expclo-id v))
		   (cond (i
			  (cond ((= i id)
				 (format t "Hacking ~s ~s ~%" x i)
				 (multiple-value-setq (x bdgs)
						      (safe-varsubst x i bdgs))
				 (cond ((has-qvars x)
					(setq x '*unbound*)))))))
		   (push (list (Varbdg-varname b)
			       x)
			 res))))
      (nreverse res)))

(defun bindings-vals (vars id bdgs)
     (let ((sb (bindings-subst bdgs id)))
         (mapcar #'(lambda (v)
		      (let ((p (assoc v sb :test #'eq)))
			 (cond ((and p (not (eq (cadr p) '*unbound*)))
				(cadr p))
			       (t
				(error "Variable not bound in bindings: ~s ~s"
				       v id)))))
		 vars)))

(defun one-bindings-vals (vars id bdgs-set allow-others)
   (cond ((null bdgs-set)
	  (error "No answer when trying to find value of ~s ~s"
		 vars id))
	 (t
	  (let ((vl (bindings-vals vars id (car bdgs-set))))
	     (cond ((and (not allow-others)
			 (not (null (cdr bdgs-set))))
		    (cerror "I will ignore them"
			    "Unexpected extra values for ~s ~s in bindings list: ~s"
			    vars id (cdr bdgs-set))))
	     vl))))

(defun bindings-val (var id bdgs)
  (let ((b (uvar-lookup var id bdgs)))
     (cond ((and b (Varbdg-val b))
	    (let ((cl (Varbdg-val b)))
	       (cond ((Expclo-id cl)
		      (let ((v (unsafe-varsubst (Expclo-skel cl)
						(Expclo-id cl)
						bdgs)))
			 (cond ((has-qvars v)
				(error "Variable not bound to constant in bindings: ~s ~s (= ~s)"
				       var id v))
			       (t v))))
		     (t (Expclo-skel cl)))))
	   (t
	    (error "Variable not bound in bindings: ~s ~s"
		   var id)))))
