;-*- Mode: Common-lisp; Package: opt; Readtable: ytools; -*-
(in-package :opt)
;;; $Id: varsubst.lisp,v 1.14 2006/10/30 21:49:22 dvm Exp $

(depends-on %module/ ytools)

(depends-on :at-run-time %langutils/ symboid)

(depends-on :at-run-time %opt/ unify)

;;; Utilities for cleaning up envs and formulas containing variables
;;; bound in those envs.

(eval-when (:compile-toplevel :load-toplevel :execute :slurp-toplevel)
   (export '(safe-varsubst unsafe-varsubst varsubst-to-constant
	     env-elim bdgenvs-combine ids-env-elim
	     bdgenv-change-id varbdg-change-id varid-convert
	     env-simplify)))

;;; This substitutes as far as possible, and as a consequence may have to
;;; return a revised env
(defun safe-varsubst (pat id alist)
   (cond (id
	  (subst-and-convert pat id id alist))
	 (t
	  (values pat alist))))

(defun subst-and-convert (pat subid destid alist)
   (cond ((is-Qvar pat)
          (cond ((eq (Qvar-sym pat) '_)
                 pat)
                (t
                 (let ((p (uvar-lookup (Qvar-sym pat) subid alist)))
                    (cond (p
                           (let ((val (Varbdg-val p)))
                              (subst-and-convert (Expclo-skel val)
                                                 (Expclo-id val)
                                                 destid alist)))
                          (t
                           (var-convert pat subid destid alist)))))))
	 ((atom pat)
	  (values pat alist))
	 (t
	  (multiple-value-bind (p1 a1)
			       (subst-and-convert
				  (car pat) subid destid alist)
	     (multiple-value-bind (p2 a2)
				  (subst-and-convert
				      (cdr pat) subid destid a1)
		(values (cons-if-new p1 p2 pat)
			a2))))))

;;; Return a version of 'env' in which all the bindings are of variables
;;; with given 'id'.
(defun env-simplify (env id)
   (cond ((not id) (empty-env))
	 (t
	  (repeat :for ((b :in env)
			(work-env env)
			:collector new-env)
	   :result (make-Bdgenv new-env)
	   :when (= (Varbdg-id b) id)
	   :within
	      (let ((val (Expclo-skel (Varbdg-val b)))
		    (valid (Expclo-id (Varbdg-val b))))
		 (cond ((not valid)
			(:continue
			 :collect b))
		       (t
			(multi-let (((val new-work-env)
				     (exp-env-elim
				        val valid id
					work-env)))
			   (:continue
			       (!= work-env new-work-env)
			    :collect (make-Varbdg
				        (Varbdg-varname b)
					id
					(make-Expclo val id)))))))))))

; This is for debugging only, because it builds patterns with inconsistent
; varids.
; Actually, it is safe in the case where the value returned contains no
; vars.
(defun unsafe-varsubst (pat id alist)
   (cond ((is-Qvar pat)
	  (let ((p (uvar-lookup (Qvar-sym pat) id alist)))
	     (cond (p
		    (let ((val (Varbdg-val p)))
		       (cond ((Expclo-id val)
			      (unsafe-varsubst (Expclo-skel val)
					       (Expclo-id val)
					       alist))
			     (t (Expclo-skel val)))))
		   (t pat))))
	 ((atom pat) pat)
	 (t
	  (let ((p1 (unsafe-varsubst (car pat) id alist))
		(p2 (unsafe-varsubst (cdr pat) id alist)))
	     (cons-if-new p1 p2 pat)))))

;;; Better
(defun varsubst-to-constant (pattern id alist)
   (letrec ((do-it (pat id alist)
	       (cond ((is-Qvar pat)
		      (let ((p (uvar-lookup (Qvar-sym pat) id alist)))
			 (cond (p
				(let ((val (Varbdg-val p)))
				   (cond ((Expclo-id val)
					  (varsubst-to-constant
					      (Expclo-skel val)
					      (Expclo-id val)
					      alist))
					 (t (Expclo-skel val)))))
			       (t
				(signal-problem varsubst-to-constant
				   "Variable " pat "[" id "] remaining in "
				    pattern
				      :% " after attempted varsubst"
				        " with [" id "] "
				      :% alist
				      (:proceed
				       "I'll proceed with variable where"
                                       " constant was expected"))))))
		     ((atom pat) pat)
		     (t
		      (let ((p1 (varsubst-to-constant (car pat) id alist))
			    (p2 (varsubst-to-constant (cdr pat) id alist)))
			 (cons-if-new p1 p2 pat))))))
      (do-it pattern id alist)))

;; Even less legal, but clearer.
(defun debug-varsubst (pat id alist)
   (cond ((is-Qvar pat)
	  (let ((p (uvar-lookup (Qvar-sym pat) id alist)))
	     (cond (p
		    (let ((val (Varbdg-val p)))
		       (cond ((Expclo-id val)
			      (let ((r (debug-varsubst (Expclo-skel val)
						       (Expclo-id val)
						       alist)))
				 (cond ((has-qvars r)
					(make-Expclo r (Expclo-id val)))
				       (t r))))
			     (t (Expclo-skel val)))))
		   (t pat))))
	 ((atom pat) pat)
	 (t
	  (let ((p1 (debug-varsubst (car pat) id alist))
		(p2 (debug-varsubst (cdr pat) id alist)))
	     (cons-if-new p1 p2 pat)))))

;;; Convert every binding that uses 'source-id' to 'dest-id'
(defun env-varid-convert (env source-id dest-id)
   (cond ((not (= source-id dest-id))
          ;;; Tricky because 'env' changes as we go through its bindings
	  (repeat :for ((bl = (Bdgenv-bdgs env) :then (rest bl)))
           :until (null bl)
	     (let* ((b (first bl))
                    (val (Varbdg-val b)))
		(let ((skel (Expclo-skel val))
		      (id (Expclo-id val)))
		   (cond ((and id (= id source-id))
                          (multi-let (((subst-skel subst-env)
                                       (varid-convert
                                          skel source-id dest-id env)))
                             (cond ((not (eq subst-skel skel))
                                    (!= env
                                        (cons (make-Varbdg
                                                 (Varbdg-varname b)
                                                 (Varbdg-id b)
                                                 (make-Expclo
                                                    subst-skel dest-id))
                                              (remove1q
                                                 b subst-env)))))))))))))
   env)

;;; We desire to combine x with another expression whose free vars
;;; have id dest-id.  To do so, we have to convert x from an expression
;;; whose free vars have source-id.  
;;; Returns new expression plus revised env
;;; x can be a list of expressions; the car is treated the same as
;;; other elements.
(defun varid-convert (x source-id dest-id env)
   (cond ((or (not source-id)
	      (eql source-id ground-id*)		; no free vars; trivial
	      (eql source-id dest-id)
	      (null env))
	  (values x env))
	 (t
	  (vars-id-convert x source-id dest-id env))))

(defun vars-id-convert (subval subid destid e)
   (cond ((or (not subid) (= subid destid))
	  (values subval e))
	 ((is-Qvar subval)
	  (let ((b (uvar-lookup (Qvar-sym subval) subid e)))
	     (cond (b
		    (let ((v (Varbdg-val b)))
		       (let ((skel (Expclo-skel v))
			     (id (Expclo-id v)))
			  (vars-id-convert skel id destid e))))
		   (t
		    (var-convert subval subid destid e)))))
	 ((atom subval)
	  (values subval e))
	 (t
	  (do ((xl subval (cdr xl))
	       (x) 
	       (res '()))
	      ((null xl)
	       (values (nreverse res) e))
	      (setq x (car xl))
	      (multiple-value-bind
	                     (v newenv)
			     (vars-id-convert x subid destid e)
	         (setf res (cons v res))
		 (setq e newenv))))))

(defun var-convert (subval subid destid e)
  (cond ((= subid destid)
	 (values subval e))
	((> subid destid)
	 (let ((newsym (new-var-sym)))
	    (let ((newvar (make-Qvar newsym '())))
	       (let ((newenv
			(create-new-bdg 
			   (Qvar-sym subval)
			   newvar
			   subid destid
			   nil e)))
		   (values newvar newenv)))))
	(t
	 (let ((bb (uvar-lookup-backwards
		      (Qvar-sym subval)
		      subid destid e)))
	    (cond (bb
		   (values (make-Qvar
			      (Varbdg-varname
				 bb)
			      '())
			   e))
		  (t
		   (new-dummy-var
		      destid subval subid e)))))))

;;; Return a version of 'exp' in which no free variable has a binding
;;; in 'env.'  (So 'env' can be discarded if we don't care about
;;; any other variables it might bind.)
(defun env-elim (exp id env)
   (nth-value 0 (exp-env-elim exp id id env)))

;;; This generates envs that violate the prohibition on earlier varid's
;;; being bound to later ones, but it doesn't matter, because these
;;; envs will be discarded
(defun exp-env-elim (subexp subid destid alist)
   (cond ((is-Qvar subexp)
	  (let ((b (uvar-lookup (Qvar-sym subexp) subid alist)))
	     (cond (b
		    (let ((val (Expclo-skel (Varbdg-val b)))
			  (valid (Expclo-id (Varbdg-val b))))
		       (exp-env-elim val valid destid alist)))
		   ((= subid destid)
		    (values subexp alist))
		   (t
		    (let ((newsym (new-var-sym)))
		       (let ((newvar (make-Qvar newsym '())))
			  (let ((newenv
				   (create-new-bdg 
				      (Qvar-sym subexp)
				      newvar
				      subid destid
				      nil alist)))
			      (values newvar newenv))))))))
	 ((atom subexp)
	  (values subexp alist))
	 (t
	  (multiple-value-bind (p1 a1)
			       (exp-env-elim
				  (car subexp) subid destid alist)
	     (multiple-value-bind (p2 a2)
				  (exp-env-elim
				      (cdr subexp) subid destid a1)
		(values (cons-if-new p1 p2 subexp)
			a2))))))

; Discard all bindings with the given id or later (greater)
;; Except optimist-vars, which must percolate all the way back to the top.
(defun bdgenv-contract (e discard)
   (cond ((null e) '())
         (t
          (let ((r (bdgenv-contract (cdr e) discard)))
             (cond ((and (>= (Varbdg-id (car e)) discard)
			 (or (not optimist-fn*)
			     (not (is-Optimist-var
				     (Varbdg-varname (car e))))))
                    r)
                   (t
                    (cons-if-new (car e) r e)))))))

(defun bdgenv-wrt-exp-simplify (env exp id)
   (bdgenv-simplify env (nodupq (raw-uvars exp)) id))

(defun bdgenv-simplify (env qvar-syms id)
   (let ((sought (<# (\\ (sym) (make-Varbdg sym id nil))
		     qvar-syms))
	 (tried !())
	 (new-bdgs !()))
      (repeat
       :until (null sought)
          (let ((var (Varbdg-varname (car sought)))
		(id (Varbdg-id (car sought))))
	     (!= tried (cons (car sought) *-*))
	     (!= sought (cdr *-*))
             (let ((b (uvar-lookup var id env)))
		(cond (b
		       (!= new-bdgs (cons b *-*))
		       (let ((val (Varbdg-val b)))
			  (let ((skel (Expclo-skel val))
				(id (Expclo-id val)))
			     (cond (id
				    (let ((vl (raw-uvars skel)))
				       (repeat :for ((v :in vl))
					  (cond ((not (or (uvar-lookup
							     v id sought)
							  (uvar-lookup
							     v id tried)))
						 (!= sought
						     (cons (make-Varbdg
							      v id nil)
							   *-*))))))))))))))
	:result (make-Bdgenv new-bdgs))))

; PAT may contain variables with DISCARD or higher.  Eliminate them,
; replacing them with new variables with id REPLACE instead.
(defun subst-away (pat e id discard replace)
   (cond ((is-Qvar pat)
          (let ((p (uvar-lookup (Qvar-sym pat) id e)))
             (cond ((and p (Varbdg-val p))
                    (let ((val (Varbdg-val p)))
                       (cond ((Expclo-id val)
                              (subst-away (Expclo-skel val) e (Expclo-id val)
                                          discard replace))
                             (t (values (Expclo-skel val) e)))))
                   ((= id replace)
                    (values pat e))
                   ((< id replace)
                    (multiple-value-bind (x e)
					 (new-dummy-var
                                            replace
                                            pat
                                            id
                                            e)
                       (values x e)))
                   (t
                    (let ((newvar (new-var-sym)))
                       (let ((e (cons (make-Varbdg
					    (Qvar-sym pat) id
					    (make-Expclo
						  (make-Qvar newvar '())
						  replace))
                                      (cons (make-Varbdg
                                                  newvar replace nil)
                                            e))))
                          (values (make-Qvar newvar '())
                                  e)))))))
         ((atom pat) (values pat e))
         (t
          (multiple-value-bind (a e1)
			       (subst-away (car pat) e id discard replace)
             (multiple-value-bind (d e2)
				  (subst-away (cdr pat) e1 id discard replace)
                (values (cons-if-new a d pat)
                        e2))))))

;; Simpler, faster interface to 'expression-car-cdr' for the usual case.
(defun e-car-cdr (exp id env)
   (cond ((and (consp exp) (not (is-Qvar exp)))
	  (values (car exp) (cdr exp) env))
	 (t
	  (multi-let (((_ a d e)
		       (expression-car-cdr exp id env true)))
	     (values a d e)))))

;;;; Get the car & cdr  of an expression.  In some cases, this may require
;;;; augmenting the environment, so return the new env.
;;;; If insist=false, then
;;;; first value returned is true if successful, false if exp is atomic
;;;; even after variable substitution.
;;;; Second value is atomic expression that results from substituting as
;;;; as possible.
;;;; If insist=true, then the same situation results in an error.
(defun expression-car-cdr (exp id env insist)
   (labels ((pursue (x subid)
	       (cond ((is-Qvar x)
		      (let ((p (uvar-lookup (Qvar-sym x) subid env)))
			 (cond (p
				(let ((val (Varbdg-val p)))
				   (pursue (Expclo-skel val)
					   (Expclo-id val))))
			       (insist
				(cerror
				   "Expression has no car and cdr ~s"
				   x))
			       (t
				(values false x nil env)
				))))
		     ((consp x)
		      (cond ((or (not subid) (= subid id))
			     (values true (car x) (cdr x) env))
			    (t
			     (multiple-value-bind
					(newx newenv)
					(varid-convert x subid id env)
				(values true (car newx) (cdr newx)
					newenv)))))
		     (insist
		      (error "Expression has no car and cdr ~s" x))
		     (t
		      (values false x nil env)))))
       (pursue exp id)))

;; Substitutes for mvars in pat, getting values from alist. 
; *Not guaranteed to substitute to any particular depth,* if
; going too far would cause it to insert variables with the wrong
; varid.
(defun varsubst (pat id alist)
   (cond ((null alist) pat)
	 (t (varsubst1 pat id alist))))

(defun varsubst1 (pat id alist)
   (cond ((is-Qvar pat)
	  (let ((p (uvar-lookup (Qvar-sym pat) id alist)))
	     (cond (p
		    (let ((val (Varbdg-val p)))
		       (cond ((Expclo-id val)
                              (multiple-value-bind (x hasvars)
						   (pat-flatten
						      (Expclo-skel val)
						      (Expclo-id val)
						      alist)
                                 (cond (hasvars pat)
                                       (t x))))
			     (t (Expclo-skel val)))))
		   (t pat))))
	 ((atom pat) pat)
	 (t
	  (let ((p1 (varsubst1 (car pat) id alist))
		(p2 (varsubst1 (cdr pat) id alist)))
	     (cons-if-new p1 p2 pat)))))

;; Try to merge env1 and env2, being careful about vars
;; bound in both.  Returns < ok, newenv >.  If envs are inconsistent,
;; first value is false.
(defun bdgenvs-combine (env1 env2)
   (cond ((null env1) (values true env2))
	 ((null env2) (values true env1))
	 (t
	  (let ((combined-env env1))
	     (repeat :for ((bdg2 :in env2)
			   ok exclo var id)
	      :result (values true combined-env)
	         (!= var (Varbdg-varname bdg2))
		 (!= id (Varbdg-id bdg2))
	         (!= exclo (Varbdg-val bdg2))
	         (!= < ok combined-env >
		     (var-unify var (Expclo-skel exclo)
				id (Expclo-id exclo)
				combined-env))
	       :while ok
	       :else result (values false nil))))))
				
;;; Eliminate all references to the given ids from env.
;;; All elements of 'ids' are Varid's (not false).
(defun ids-env-elim (ids env)
   (letrec ((id-is-bad (id) (member id ids))
	    (bdg-is-bad (b)
	       (cond ((id-is-bad (Varbdg-id b))
		      ;; We don't care about bindings of bad vars
		      false)
		     (t
		      ;; We do care about bindings of good vars to bad
		      (let ((val (Varbdg-val b)))
			 (let ((val-id (Expclo-id val)))
			    (and val-id (id-is-bad val-id)))))))
	    (bdg-has-bad-id (b)
	       (or (id-is-bad (Varbdg-id b))
		   (bdg-is-bad b))))
;;;;      (out (:i> 3)
;;;;	   "env = "
;;;;	   (:e (<# (\\ (b)
;;;;		      (:o "[" b "/" (:q ((bdg-has-bad-id b)
;;;;					 "bad")
;;;;					(t "good"))
;;;;			  "]" :%))
;;;;		   env))
;;;;	   :%)
      (cond ((and (not (null ids))
		  (exists (b :in env)
		     (bdg-has-bad-id b)))
	     (let ((var-repl-tab !((Lrcd Symbol Symbol))))
	            ;; -- replacements for all vars with bad ids
                (let ((repl-id
		         (</ (\\ (low-id b)
				(cond ((bdg-is-bad b)
				       (cond (low-id
					      (min low-id (Varbdg-id b)))
					     (t (Varbdg-id b))))
				      (t low-id)))
			     false env)))
		   (repeat :for ((bdgs-to-check = env :then check-bdgs)
				 (check-bdgs = !() :then :again)
				 (good-bdgs = !())
				 (curenv = env
					 :then (append check-bdgs curenv)))
		      (repeat :for ((b :in bdgs-to-check))
			 (cond ((bdg-is-bad b)
				(let ((val (Varbdg-val b)))
				   (let ((val-id (Expclo-id val)))
				      (multi-let (((newskel new-curenv)
						   (safe-varsubst
						       (Expclo-skel val)
						       val-id
						       curenv)))
					 (!= check-bdgs
					     (nconc (ldiff new-curenv curenv)
						    *-*))
					 (!= curenv new-curenv)
					 (!= good-bdgs
					     (cons (make-Varbdg
						      (Varbdg-varname b)
						      (Varbdg-id b)
						      (skel-subst
						         newskel))
						   *-*))))))))
;;;;		      (out "bdgs-to-check = "
;;;;			   (:e (<# (\\ (b)
;;;;				      (:o "[" b "/" (:q ((bdg-has-bad-id b)
;;;;							 "bad")
;;;;							(t "good"))
;;;;					  "]" :%))
;;;;				   env))
;;;;			   :% "check-bdgs = " check-bdgs
;;;;			   :% "good-bdgs = " good-bdgs
;;;;			   :%)
		    :until (null check-bdgs)
		    :result (nconc good-bdgs
				   (<? (\\ (b)
					  (not (bdg-has-bad-id b)))
				       curenv))
		    :where
		      (skel-subst (skel)
			 (let ((found-var false))
			    (letrec ((do-subst (x)
					(cond ((is-Qvar x)
					       (!= found-var true)
					       (var-replace x))
					      ((atom x) x)
					      (t
					       (cons-if-new
						  (do-subst (car x))
						  (do-subst (cdr x))
						  x)))))
			       (let ((newskel (do-subst skel)))
				  (make-Expclo
				     newskel
				     (cond (found-var repl-id)
					   (t false)))))))
		      (var-replace (qvar)
			 (let ((p (assq (Qvar-sym qvar) var-repl-tab)))
			    (cond ((not p)
				   (!= p (tuple (Qvar-sym qvar)
						(new-var-sym)))
				   (!= var-repl-tab
				       (cons p *-*))))
			    (make-Qvar (cadr p) !())))))))
	    (t env))))

(defun bdgenv-change-id (env old-id new-id)
   (cond ((and old-id new-id)
	  (cond ((= old-id new-id)
		 env)
		(t
		 (<# (\\ (b)
			(varbdg-change-id b old-id new-id))
		     env))))
	 ((or (null env)
	      (not (or old-id new-id)))
	  env)
	 (t
	  (signal-problem bdgenv-change-id
	     "Incoherent call to bdgenv-change-id:"
	     :% "  old-id = " old-id "  new-id = " new-id
	     :% "  env = " env))))

(defun varbdg-change-id (b oldid newid)
   (make-Varbdg
      (Varbdg-varname b)
      (cond ((= (Varbdg-id b) oldid)
	     newid)
	    (t (Varbdg-id b)))
      (let ((val (Varbdg-val b)))
	 (let ((skel (Expclo-skel val))
	       (skelid (Expclo-id val)))
	    (cond ((or (not skelid)
		       (not (= skelid oldid)))
		   val)
		  (t
		   (make-Expclo skel newid)))))))

